package cn.com.poc.agent_application.rest.impl;

import cn.com.poc.agent_application.aggregate.AgentApplicationInfoService;
import cn.com.poc.agent_application.convert.AgentApplicationInfoConvert;
import cn.com.poc.agent_application.dto.*;
import cn.com.poc.agent_application.entity.BizAgentApplicationInfoEntity;
import cn.com.poc.agent_application.entity.BizAgentApplicationLargeModelListEntity;
import cn.com.poc.agent_application.entity.CreateAgentTitleAndDescEntity;
import cn.com.poc.agent_application.query.AgentApplicationInfoQueryCondition;
import cn.com.poc.agent_application.rest.AgentApplicationInfoRest;
import cn.com.poc.agent_application.service.BizAgentApplicationInfoService;
import cn.com.poc.agent_application.service.BizAgentApplicationLargeModelListService;
import cn.com.poc.agent_application.service.BizAgentApplicationPublishService;
import cn.com.poc.common.constant.CommonConstant;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.common.utils.JsonUtils;
import cn.com.poc.knowledge.aggregate.KnowledgeService;
import cn.com.poc.support.security.oauth.entity.UserBaseEntity;
import cn.com.poc.thirdparty.resource.demand.ai.common.domain.Tool;
import cn.com.poc.thirdparty.resource.demand.ai.function.LargeModelFunctionEnum;
import cn.com.yict.framemax.core.exception.BusinessException;
import cn.com.yict.framemax.data.model.PagingInfo;
import cn.hutool.core.collection.ListUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component
public class AgentApplicationInfoRestImpl implements AgentApplicationInfoRest {

    @Resource
    private BizAgentApplicationInfoService bizAgentApplicationInfoService;

    @Resource
    private AgentApplicationInfoService agentApplicationInfoService;

    @Resource
    private BizAgentApplicationPublishService bizAgentApplicationPublishService;

    @Resource
    private BizAgentApplicationLargeModelListService bizAgentApplicationLargeModelListService;

    @Resource
    private KnowledgeService knowledgeService;

    public List<AgentApplicationInfoDto> getListByMember(AgentApplicationInfoSearchDto dto, PagingInfo pagingInfo) throws Exception {
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        Long userId = userBaseEntity.getUserId();
        AgentApplicationInfoQueryCondition condition = new AgentApplicationInfoQueryCondition();
        condition.setMemberId(userId);
        if (!StringUtils.isEmpty(dto.getQuery())) {
            condition.setQuery(dto.getQuery());
        }
        if (StringUtils.isNotBlank(dto.getPublishStatus())) {
            condition.setAgentPublishStatus(dto.getPublishStatus());
        }
        List<BizAgentApplicationInfoEntity> bizAgentApplicationInfoEntities = bizAgentApplicationInfoService.agentApplicationInfoQuery(condition, pagingInfo);
        List<AgentApplicationInfoDto> result = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(bizAgentApplicationInfoEntities)) {
            result = bizAgentApplicationInfoEntities.stream().map(AgentApplicationInfoConvert::entityToDto).collect(Collectors.toList());
        }
        return result;

    }

    public AgentApplicationInfoDto saveOrUpdate(AgentApplicationInfoDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getBaseInfo());
        Assert.notNull(dto.getCommConfig());
        Assert.notNull(dto.getCommModelConfig());
        Assert.notNull(dto.getKnowledgeConfig());
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        Long userId = userBaseEntity.getUserId();
        dto.getBaseInfo().setMemberId(userId.intValue());
        BizAgentApplicationInfoEntity entity = AgentApplicationInfoConvert.dtoToEntity(dto);
        return AgentApplicationInfoConvert.entityToDto(StringUtils.isEmpty(entity.getAgentId()) ?
                bizAgentApplicationInfoService.save(entity) : bizAgentApplicationInfoService.update(entity));
    }


    public AgentApplicationInfoDto update(AgentApplicationInfoDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getBaseInfo());
        Assert.notNull(dto.getBaseInfo().getAgentId(), "id不能为空");
        Assert.notNull(dto.getCommConfig());
        Assert.notNull(dto.getCommModelConfig());
        Assert.notNull(dto.getKnowledgeConfig());
        BizAgentApplicationInfoEntity entity = AgentApplicationInfoConvert.dtoToEntity(dto);
        return AgentApplicationInfoConvert.entityToDto(bizAgentApplicationInfoService.update(entity));
    }

    @Override
    public AgentApplicationInfoDto getInfo(String agentId) throws Exception {
        Assert.notNull(agentId);
        BizAgentApplicationInfoEntity entity = bizAgentApplicationInfoService.getByAgentId(agentId);
        if (entity == null) {
            throw new BusinessException("应用不存在");
        }
        return AgentApplicationInfoConvert.entityToDto(entity);
    }

    @Override
    public void delete(String agentId) throws Exception {
        Assert.notNull(agentId);
        bizAgentApplicationInfoService.deletedByAgentId(agentId);
        bizAgentApplicationPublishService.deleteByAgentId(agentId);
    }

    @Override
    public boolean updateAndPublish(AgentApplicationInfoDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getBaseInfo());
        Assert.notNull(dto.getBaseInfo().getAgentId(), "id不能为空");
        Assert.notNull(dto.getBaseInfo().getAgentTitle(), "应用名称不能为空");
        Assert.notNull(dto.getCommConfig());
        Assert.notNull(dto.getCommModelConfig());
        Assert.notNull(dto.getKnowledgeConfig());
        BizAgentApplicationInfoEntity bizAgentApplicationInfoEntity = AgentApplicationInfoConvert.dtoToEntity(dto);
        agentApplicationInfoService.updateAndPublish(bizAgentApplicationInfoEntity);
        return true;
    }

    @Override
    public void preview(AgentApplicationPreviewDto dto, HttpServletResponse httpServletResponse) throws IOException {
        Assert.notNull(dto);
        Assert.notNull(dto.getAgentId());
        Assert.notEmpty(dto.getMessages());
        try {
            String agentId = dto.getAgentId();
            BizAgentApplicationInfoEntity infoEntity = bizAgentApplicationInfoService.getByAgentId(agentId);
            if (infoEntity == null) {
                throw new BusinessException("应用不存在");
            }
            //获取知识库配置
            List<Integer> kdIds = knowledgeService.getKdIdsByKnowledgeInfoIds(infoEntity.getKnowledgeIds());

            //配置对话function
            List<Tool> tools = new ArrayList<>();
            //开启对话变量
            if (ArrayUtils.isNotEmpty(infoEntity.getVariableStructure())) {
                String functionName = "set_value_memory";
                String llmConfig = LargeModelFunctionEnum.valueOf(functionName).getFunction().getVariableStructureLLMConfig(infoEntity.getVariableStructure()).get(0);
                Tool tool = JsonUtils.deSerialize(llmConfig, Tool.class);
                tools.add(tool);
            }
            //开启长期记忆
            if (CommonConstant.YOrN.Y.equals(infoEntity.getIsLongMemory())) {
                String functionName = "set_long_memory";
                String llmConfig = LargeModelFunctionEnum.valueOf(functionName).getFunction().getLLMConfig().get(0);
                Tool tool = JsonUtils.deSerialize(llmConfig, Tool.class);
                tools.add(tool);
            }

            //调用应用服务
            agentApplicationInfoService.callAgentApplication(agentId, infoEntity.getLargeModel(), infoEntity.getUnitIds()
                    , infoEntity.getAgentSystem(), kdIds.toArray(new Integer[0]), infoEntity.getCommunicationTurn(), infoEntity.getTopP()
                    , dto.getMessages(), tools, httpServletResponse);
        } catch (Exception e) {
            httpServletResponse.setContentType("text/event-stream");
            PrintWriter writer = httpServletResponse.getWriter();
            writer.write("data: {\"code\":-1,\"message\":\"" + e.getLocalizedMessage() + "\"} \n\n");
            writer.write("data: [DONE]\n\n");
            writer.flush();
        }
    }

    @Override
    public List<BizAgentApplicationLargeModelListDto> getLargeModelList() throws Exception {
        List<BizAgentApplicationLargeModelListEntity> entities = bizAgentApplicationLargeModelListService.findByExample(new BizAgentApplicationLargeModelListEntity(), null);
        Map<String, List<BizAgentApplicationLargeModelListEntity>> map = entities.stream()
                .collect(Collectors.groupingBy(BizAgentApplicationLargeModelListEntity::getOwner));

        List<BizAgentApplicationLargeModelListDto> result = new ArrayList<>();
        for (String owner : map.keySet()) {
            BizAgentApplicationLargeModelListDto dto = new BizAgentApplicationLargeModelListDto();
            List<BizAgentApplicationLargeModelListEntity> modelListEntities = map.get(owner);
            dto.setOwner(owner);
            dto.setModels(modelListEntities.stream().map(BizAgentApplicationLargeModelListEntity::getModelNickName).collect(Collectors.toList()));
            dto.setIcon(modelListEntities.get(0).getIconUrl());
            result.add(dto);
        }
        return result;
    }

    @Override
    public BizAgentApplicationLargeModelListDto getLargeModelInfo(String query) throws Exception {
        Assert.notNull(query);
        BizAgentApplicationLargeModelListEntity bizAgentApplicationLargeModelListEntity = new BizAgentApplicationLargeModelListEntity();
        bizAgentApplicationLargeModelListEntity.setModelNickName(query);
        List<BizAgentApplicationLargeModelListEntity> entities = bizAgentApplicationLargeModelListService.findByExample(bizAgentApplicationLargeModelListEntity, null);
        if (CollectionUtils.isEmpty(entities)) {
            throw new BusinessException("模型不存在");
        }
        BizAgentApplicationLargeModelListEntity entity = entities.get(0);
        BizAgentApplicationLargeModelListDto result = new BizAgentApplicationLargeModelListDto();
        result.setModels(ListUtil.toList(entity.getModelNickName()));
        result.setOwner(entity.getOwner());
        result.setIcon(entity.getIconUrl());
        return result;
    }

    @Override
    public void createAgentSystem(AgentApplicationGCDto dto, HttpServletResponse response) throws Exception {
        Assert.notNull(dto.getInput(), "输入不能为空");
        agentApplicationInfoService.createAgentSystem(dto.getInput(), response);
    }

    @Override
    public List<String> createFeaturedQuestions(AgentApplicationGCDto dto) throws Exception {
        return agentApplicationInfoService.createFeaturedQuestions(dto.getAgentTitle(), dto.getAgentDesc());
    }

    @Override
    public String createPreamble(AgentApplicationGCDto dto) throws Exception {
        return agentApplicationInfoService.createPreamble(dto.getAgentTitle(), dto.getAgentDesc(), dto.getAgentSystem());
    }

    @Override
    public String createAgentApplicationAvatar(AgentApplicationGCDto dto) throws Exception {
        Assert.isTrue(StringUtils.isNotBlank(dto.getAgentDesc()) || StringUtils.isNotBlank(dto.getAgentTitle()), "请输入标题或者描述再生成应用头像");
        return agentApplicationInfoService.createAgentIcon(dto.getAgentTitle(), dto.getAgentDesc());
    }

    @Override
    public AgentApplicationGCDto createAgentTitleAndDesc(AgentApplicationGCDto dto) throws Exception {
        Assert.notNull(dto.getInput(), "输入不能为空");
        CreateAgentTitleAndDescEntity entity = agentApplicationInfoService.createAgentTitleAndDesc(dto.getInput());
        if (null == entity) {
            throw new BusinessException("生成失败,请稍后重试");
        }
        AgentApplicationGCDto result = new AgentApplicationGCDto();
        result.setAgentTitle(entity.getAgentTitle());
        result.setAgentDesc(entity.getAgentDesc());
        return result;
    }
}