package cn.com.poc.common.service.impl;

import cn.com.poc.common.service.RedisService;
import cn.com.poc.common.utils.JsonUtils;
import cn.com.yict.framemax.core.exception.BusinessException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisZSetCommands;
import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Component
public class RedisServiceImpl implements RedisService {

    @Resource(name = "redisTemplate")
    private RedisTemplate redisTemplate;

    /****************** common start ****************/
    /**
     * 指定缓存失效时间
     *
     * @param key  键
     * @param time 时间(秒)
     * @return
     */
    @Override
    public boolean expire(String key, long time) {
        try {
            if (time > 0) {
                redisTemplate.expire(key, time, TimeUnit.SECONDS);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 根据key 获取过期时间
     *
     * @param key 键 不能为null
     * @return 时间(秒) 返回0代表为永久有效
     */
    @Override
    public long getExpire(String key) {
        return redisTemplate.getExpire(key, TimeUnit.SECONDS);
    }

    /**
     * 判断key是否存在
     *
     * @param key 键
     * @return true 存在 false不存在
     */
    @Override
    public boolean hasKey(String key) {
        try {
            return redisTemplate.hasKey(key);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 删除缓存
     *
     * @param key 可以传一个值 或多个
     */
    @Override
    public void del(String... key) {
        if (key != null && key.length > 0) {
            if (key.length == 1) {
                redisTemplate.delete(key[0]);
            } else {
//                redisTemplate.delete(CollectionUtils.arrayToList(key));
            }
        }
    }

    /**
     * 根据key的前缀模糊删除数据
     *
     * @param keyPre redis中key的前缀
     */
    @Override
    public void delByPre(String keyPre) {

        keyPre = keyPre + "**";
        Set<String> keys = redisTemplate.keys(keyPre);
        if (CollectionUtils.isNotEmpty(keys)) {
            redisTemplate.delete(keys);
        }

    }

    /****************** common end ****************/


    /****************** String start ****************/

    /**
     * 普通缓存获取
     *
     * @param key 键
     * @return 值
     */
    @SuppressWarnings("unchecked")
    @Override
    public Object get(String key) {
        return key == null ? null : redisTemplate.opsForValue().get(key);
    }

    /**
     * 普通缓存获取 指定序列化对象
     *
     * @param key
     * @param clazz
     * @return
     */
    @Override
    public <T> T get(String key, Class<T> clazz) {
        if (StringUtils.isEmpty(key)) {
            return null;
        }
        Object obj = redisTemplate.opsForValue().get(key);

        return JsonUtils.convert(obj, clazz);
    }

    /**
     * 普通缓存放入
     *
     * @param key   键
     * @param value 值
     * @return true成功 false失败
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean set(String key, Object value) {
        try {
            redisTemplate.opsForValue().set(key, value);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 普通缓存放入并设置时间
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒) time要大于0 如果time小于等于0 将设置无限期
     * @return true成功 false 失败
     */
    @Override
    public boolean set(String key, Object value, long time) {
        try {
            if (time > 0) {
                redisTemplate.opsForValue().set(key, value, time, TimeUnit.SECONDS);
            } else {
                set(key, value);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 递增
     *
     * @param key   键
     * @param delta 要增加几(大于0)
     * @return
     */
    @Override
    public long incr(String key, long delta) {
        if (delta < 0) {
            throw new RuntimeException("递增因子必须大于0");
        }
        try {
            return redisTemplate.opsForValue().increment(key, delta);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 递减
     *
     * @param key   键
     * @param delta 要减少几(小于0)
     * @return
     */
    @Override
    public long decr(String key, long delta) {
        if (delta < 0) {
            throw new RuntimeException("递减因子必须大于0");
        }
        try {
            return redisTemplate.opsForValue().increment(key, -delta);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }
    /****************** String end ****************/


    /****************** Map start ****************/

    /**
     * HashGet
     *
     * @param key  键 不能为null
     * @param item 项 不能为null
     * @return 值
     */
    @Override
    public Object hget(String key, String item) {
        return redisTemplate.opsForHash().get(key, item);
    }

    /**
     * 获取hashKey对应的所有键值
     *
     * @param key 键
     * @return 对应的多个键值
     */
    @Override
    public Map<Object, Object> hmget(String key) {
        return redisTemplate.opsForHash().entries(key);
    }

    /**
     * HashSet
     *
     * @param key 键
     * @param map 对应多个键值
     * @return true 成功 false 失败
     */
    @Override
    public boolean hmset(String key, Map<String, Object> map) {
        try {
            redisTemplate.opsForHash().putAll(key, map);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }


    /**
     * 批量更新hash
     *
     * @param data 多个hash
     * @param time 过期时间（秒），小于0时为不过期
     * @return 是否成功操作
     */
    @Override
    public boolean hmSetBatch(Map<String, Map<String, Object>> data, long time) {

        //参数判空
        if (MapUtils.isEmpty(data)) {
            return true;
        }

        //管道批处理
        redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                RedisOperations<String, Object> redisOperations = (RedisOperations<String, Object>) operations;
                for (String key : data.keySet()) {
                    //批量更新
                    redisOperations.opsForHash().putAll(key, data.get(key));
                    //有设置过期时间时过期
                    if (time > 0) {
                        redisOperations.expire(key, time, TimeUnit.SECONDS);
                    }
                }
                return null;
            }
        });
        return true;

    }

    /**
     * 根据key前缀批量模糊批量查询hash类型的数据
     *
     * @param keyPre key前缀
     * @return 有则返回数据，无则返回null
     */
    @Override
    public Map<String, Map<String, Object>> hmDimMultiGet(String keyPre) {

        Map<String, Map<String, Object>> returnData = new HashMap<>();
        //前缀
        keyPre = keyPre + "**";
        //获取所有的key
        Set<String> keys = redisTemplate.keys(keyPre);
        if (CollectionUtils.isNotEmpty(keys)) {
            for (String key : keys) {
                //查询key的value
                Map<Object, Object> hashObject = redisTemplate.opsForHash().entries(key);
                //类型转换,使用entrySet避免了使用keySet多次get()查找的效率损耗
                Map<String, Object> hashString = new HashMap<>();
                for (Map.Entry<Object, Object> entry : hashObject.entrySet()) {
                    hashString.put((String) entry.getKey(), entry.getValue());
                }
                returnData.put(key, hashString);
            }
            return returnData;
        }
        return null;

    }

    /**
     * HashSet 并设置时间
     *
     * @param key  键
     * @param map  对应多个键值
     * @param time 时间(秒)
     * @return true成功 false失败
     */
    @Override
    public boolean hmset(String key, Map<String, Object> map, long time) {
        try {
            redisTemplate.opsForHash().putAll(key, map);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 向一张hash表中放入数据,如果不存在将创建
     *
     * @param key   键
     * @param item  项
     * @param value 值
     * @return true 成功 false失败
     */
    @Override
    public boolean hset(String key, String item, Object value) {
        try {
            redisTemplate.opsForHash().put(key, item, value);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 向一张hash表中放入数据,如果不存在将创建
     *
     * @param key   键
     * @param item  项
     * @param value 值
     * @param time  时间(秒) 注意:如果已存在的hash表有时间,这里将会替换原有的时间
     * @return true 成功 false失败
     */
    @Override
    public boolean hset(String key, String item, Object value, long time) {
        try {
            redisTemplate.opsForHash().put(key, item, value);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 删除hash表中的值
     *
     * @param key  键 不能为null
     * @param item 项 可以使多个 不能为null
     */
    @Override
    public void hdel(String key, Object... item) {
        try {
            redisTemplate.opsForHash().delete(key, item);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 判断hash表中是否有该项的值
     *
     * @param key  键 不能为null
     * @param item 项 不能为null
     * @return true 存在 false不存在
     */
    @Override
    public boolean hHasKey(String key, String item) {
        return redisTemplate.opsForHash().hasKey(key, item);
    }

    /**
     * hash递增 如果不存在,就会创建一个 并把新增后的值返回
     *
     * @param key  键
     * @param item 项
     * @param by   要增加几(大于0)
     * @return
     */
    @Override
    public double hincr(String key, String item, long by) {
        try {
            return redisTemplate.opsForHash().increment(key, item, by);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * hash递减
     *
     * @param key  键
     * @param item 项
     * @param by   要减少记(小于0)
     * @return
     */
    @Override
    public double hdecr(String key, String item, long by) {
        try {
            return redisTemplate.opsForHash().increment(key, item, -by);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }


    /****************** Map end ****************/


    /****************** Set start ****************/

    /**
     * 根据key获取Set中的所有值
     *
     * @param key 键
     * @return
     */
    @Override
    public Set<Object> sGet(String key) {
        try {
            return redisTemplate.opsForSet().members(key);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 根据value从一个set中查询,是否存在
     *
     * @param key   键
     * @param value 值
     * @return true 存在 false不存在
     */
    @Override
    public boolean sHasKey(String key, Object value) {
        try {
            return redisTemplate.opsForSet().isMember(key, value);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }


    /**
     * 将数据放入set缓存
     *
     * @param key    键
     * @param values 值 可以是多个
     * @return 成功个数
     */
    @Override
    public long sSet(String key, Object... values) {
        try {
            return redisTemplate.opsForSet().add(key, values);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 将set数据放入缓存
     *
     * @param key    键
     * @param time   时间(秒)
     * @param values 值 可以是多个
     * @return 成功个数
     */
    @Override
    public long sSetAndTime(String key, long time, Object... values) {
        try {
            Long count = redisTemplate.opsForSet().add(key, values);
            if (time > 0) {
                expire(key, time);
            }
            return count;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }


    /**
     * 获取set缓存的长度
     *
     * @param key 键
     * @return
     */
    @Override
    public long sGetSetSize(String key) {
        try {
            return redisTemplate.opsForSet().size(key);
        } catch (Exception e) {
            e.printStackTrace();
            return 0;
        }
    }


    /**
     * 移除值为value的
     *
     * @param key    键
     * @param values 值 可以是多个
     * @return 移除的个数
     */
    @Override
    public long setRemove(String key, Object... values) {
        try {
            Long count = redisTemplate.opsForSet().remove(key, values);
            return count;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }


    /****************** Set end ****************/

    /**
     * ZSet start
     *
     * Redis zset 和 set 一样也是string类型元素的集合,且不允许重复的成员。
     * 不同的是每个元素都会关联一个double类型的分数。
     * redis正是通过分数来为集合中的成员进行从小到大的排序。
     * zset的成员是唯一的,但分数(score)却可以重复。
     */

    /**
     * @param key   键
     * @param value 值
     * @param score 分数
     * @return
     * @apiNote ZSet存放单个元素
     */
    @Override
    public Boolean zAdd(String key, Object value, Double score) {

        try {
            return redisTemplate.opsForZSet().add(key, value, score);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }

    }

    /**
     * @param key       ZSet中键
     * @param tuplesMap ZSet中value于score的映射，tuplesMap键为ZSet中的value，值为ZSet中的score
     * @return
     * @apiNote 往ZSet中指定键批量插入数据
     */
    @Override
    public Long zBatchAdd(String key, Map<Object, Double> tuplesMap) {

        try {
            if (tuplesMap.size() == 0) {
                return 0L;
            }
            Set<ZSetOperations.TypedTuple<Object>> tuplesSet = new HashSet<>();
            for (Map.Entry<Object, Double> tuplesMapEntry : tuplesMap.entrySet()) {
                tuplesSet.add(new DefaultTypedTuple<>(tuplesMapEntry.getKey(), tuplesMapEntry.getValue()));
            }
            return redisTemplate.opsForZSet().add(key, tuplesSet);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }

    }

    /**
     * @param key   键
     * @param value ZSet中的一个或多个value
     * @return
     * @apiNote 删除ZSet中指定键的一个或多个value
     */
    @Override
    public Long zRemove(String key, Object... value) {

        try {
            return redisTemplate.opsForZSet().remove(key, value);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }

    }

    /**
     * @param key 键
     * @return
     * @apiNote 获取ZSet指定键的个数
     */
    @Override
    public Long zGetSize(String key) {

        try {
            return redisTemplate.opsForZSet().size(key);
        } catch (Exception e) {
            e.printStackTrace();
            return 0L;
        }

    }

    /**
     * @param key      键
     * @param minScore 分数范围左
     * @param maxScore 分数范围右
     * @return
     * @apiNote 获取ZSet中指定键的指定分数段的value个数
     */
    @Override
    public Long zGetCount(String key, Double minScore, Double maxScore) {

        try {
            return redisTemplate.opsForZSet().count(key, minScore, maxScore);
        } catch (Exception e) {
            e.printStackTrace();
            return 0L;
        }

    }

    /**
     * @param key   键
     * @param start 开始索引
     * @param end   结束索引
     * @return
     * @apiNote 使用索引下标获取ZSet元素, [0,-1]表示全部元素,列表对于score顺数
     */
    @Override
    public Set<RedisZSetCommands.Tuple> zRangeWithScores(String key, Integer start, Integer end) {

        try {
            /**
             * 从回调外部获取连接打开一个新连接，并在等待响应的情况下执行Redis命令-
             * 没有流水线应用于任何外部获得的连接。
             * 防止当流水线会话同步并executePipelined(…)终止时，不会在回调内收到结果值，而是在最后。
             */
            RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
            return connection.zRangeWithScores(key.getBytes(), start, end);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * @param key   键
     * @param start 开始索引
     * @param end   结束索引
     * @return
     * @apiNote 使用索引下标获取ZSet元素, [0,-1]表示全部元素,列表对于score倒数
     */
    @Override
    public Set<RedisZSetCommands.Tuple> zRevRangeWithScores(String key, Integer start, Integer end) {

        try {
            /**
             * 从回调外部获取连接打开一个新连接，并在等待响应的情况下执行Redis命令-
             * 没有流水线应用于任何外部获得的连接。
             * 防止当流水线会话同步并executePipelined(…)终止时，不会在回调内收到结果值，而是在最后。
             */
            RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
            return connection.zRevRangeWithScores(key.getBytes(), start, end);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * @param key   键
     * @param start 索引开始
     * @param end   索引结束
     * @return
     * @apiNote 使用索引下标获取ZSet元素的value值集合, [0,-1]表示全部元素
     */
    @Override
    public Set<Object> zRange(String key, Integer start, Integer end) {

        try {
            return redisTemplate.opsForZSet().range(key, start, end);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * @param key   键
     * @param start 索引开始
     * @param end   索引结束
     * @return
     * @apiNote 使用索引下标获取ZSet元素的value值集合, 并从高到低排序, [0,-1]表示全部元素
     */
    @Override
    public Set<Object> zReverseRange(String key, Integer start, Integer end) {

        try {
            return redisTemplate.opsForZSet().reverseRange(key, start, end);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * @param key   键
     * @param value ZSet中value
     * @return
     * @apiNote 获取ZSet中指定键指定value的分数
     */
    @Override
    public Double zGetScore(String key, Object value) {

        try {
            return redisTemplate.opsForZSet().score(key, value);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /****************** ZSet end ****************/

    /****************** List start ****************/

    /**
     * 获取list缓存的内容
     *
     * @param key   键
     * @param start 开始
     * @param end   结束 0 到 -1代表所有值
     * @return
     */
    @Override
    public List<Object> lGet(String key, long start, long end) {
        try {
            return redisTemplate.opsForList().range(key, start, end);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }


    /**
     * 获取list缓存的长度
     *
     * @param key 键
     * @return
     */
    @Override
    public long lGetListSize(String key) {
        try {
            return redisTemplate.opsForList().size(key);
        } catch (Exception e) {
            e.printStackTrace();
            return 0;
        }
    }


    /**
     * 通过索引 获取list中的值
     *
     * @param key   键
     * @param index 索引 index>=0时， 0 表头，1 第二个元素，依次类推；index<0时，-1，表尾，-2倒数第二个元素，依次类推
     * @return
     */
    @Override
    public Object lGetIndex(String key, long index) {
        try {
            return redisTemplate.opsForList().index(key, index);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @return
     */
    @Override
    public boolean lSet(String key, Object value) {
        try {
            redisTemplate.opsForList().rightPush(key, value);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒)
     * @return
     */
    @Override
    public boolean lSet(String key, Object value, long time) {
        try {
            redisTemplate.opsForList().rightPush(key, value);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @return
     */
    @Override
    public boolean lSet(String key, List<Object> value) {
        try {
            redisTemplate.opsForList().rightPushAll(key, value);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒)
     * @return
     */
    @Override
    public boolean lSet(String key, List<Object> value, long time) {
        try {
            redisTemplate.opsForList().rightPushAll(key, value);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 根据索引修改list中的某条数据
     *
     * @param key   键
     * @param index 索引
     * @param value 值
     * @return
     */
    @Override
    public boolean lUpdateIndex(String key, long index, Object value) {
        try {
            redisTemplate.opsForList().set(key, index, value);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 移除N个值为value
     *
     * @param key   键
     * @param count 移除多少个
     * @param value 值
     * @return 移除的个数
     */
    @Override
    public long lRemove(String key, long count, Object value) {
        try {
            Long remove = redisTemplate.opsForList().remove(key, count, value);
            return remove;
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 模糊匹配查询key
     *
     * @param key
     * @return
     */
    @Override
    public Set<String> keys(String key) {
        return redisTemplate.keys(key);
    }

}

