package cn.com.poc.knowledge.rest.impl;

import cn.com.poc.common.constant.CommonConstant;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.common.utils.FileUtils;
import cn.com.poc.common.utils.JsonUtils;
import cn.com.poc.knowledge.aggregate.KnowledgeService;
import cn.com.poc.knowledge.constant.KnowledgeConstant;
import cn.com.poc.knowledge.convert.BizKnowledgeInfoConvert;
import cn.com.poc.knowledge.dto.BizKnowledgeDocumentDto;
import cn.com.poc.knowledge.dto.BizKnowledgeInfoDto;
import cn.com.poc.knowledge.dto.TrainKnowledgeDto;
import cn.com.poc.knowledge.dto.UpsertChunkInfoDto;
import cn.com.poc.knowledge.entity.BizKnowledgeDocumentEntity;
import cn.com.poc.knowledge.entity.BizKnowledgeInfoEntity;
import cn.com.poc.knowledge.query.KnowledgeInfosQueryCondition;
import cn.com.poc.knowledge.query.KnowledgeInfosQueryItem;
import cn.com.poc.knowledge.rest.KnowledgeRest;
import cn.com.poc.knowledge.service.BizKnowledgeInfoService;
import cn.com.poc.support.security.oauth.entity.UserBaseEntity;
import cn.com.poc.thirdparty.resource.demand.ai.entity.knowledge.GetKnowledgeChunkInfoResult;
import cn.com.yict.framemax.core.exception.BusinessException;
import cn.com.yict.framemax.data.model.PagingInfo;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.TypeReference;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Component
public class KnowledgeRestImpl implements KnowledgeRest {

    @Resource
    private KnowledgeService knowledgeService;

    @Resource
    private BizKnowledgeInfoService bizKnowledgeInfoService;

    @Override
    public List<BizKnowledgeDocumentDto> uploadDocument(MultipartFile[] documentFiles) throws Exception {
        Assert.notEmpty(documentFiles);
        Assert.isTrue(documentFiles.length <= 5, "Uploaded files can be no larger than five");
        List<BizKnowledgeDocumentEntity> entities = knowledgeService.uploadDocument(documentFiles);
        List<BizKnowledgeDocumentDto> result = entities.stream().map(entity -> {
            BizKnowledgeDocumentDto dto = new BizKnowledgeDocumentDto();
            BeanUtil.copyProperties(entity, dto);
            dto.setDocumentSize(FileUtils.formatFileSize(entity.getDocumentSize()));
            return dto;
        }).collect(Collectors.toList());
        return result;
    }

    @Override
    public Boolean delDocument(Integer knowledgeInfoId, Integer kdId) {
        Assert.notNull(knowledgeInfoId);
        Assert.notNull(kdId);
        return knowledgeService.delDocument(knowledgeInfoId, kdId);
    }

    @Override
    public Boolean batchDelDocument(Integer knowledgeInfoId, List<Integer> kdId) {
        Assert.notEmpty(kdId);
        Assert.notNull(knowledgeInfoId);
        boolean result = true;
        for (Integer id : kdId) {
            result = result && knowledgeService.delDocument(knowledgeInfoId, id);
        }
        return result;
    }

    @Override
    public Boolean renameDocument(Integer kdId, String rename) throws Exception {
        Assert.notNull(kdId);
        Assert.notEmpty(rename);
        return knowledgeService.renameDocument(kdId, rename);
    }

    @Override
    public String enableKnowledge(Integer kdId) throws Exception {
        Assert.notNull(kdId);
        return knowledgeService.enableKnowledge(kdId);
    }

    @Override
    public Boolean trainKnowledge(TrainKnowledgeDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getKnowledgeInfoId());
        Assert.notNull(dto.getKdIds());
        Assert.notNull(dto.getSegmentationConfig());
        return knowledgeService.trainKnowledge(dto.getKnowledgeInfoId(), dto.getKdIds(), dto.getSegmentationConfig());
    }

    @Override
    public List<BizKnowledgeDocumentDto> searchDocuments(String search, String trainStatus, Integer knowledgeInfoId, PagingInfo pagingInfo) throws Exception {
        Assert.notNull(knowledgeInfoId);
        List<BizKnowledgeDocumentDto> result = new ArrayList<>();
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = bizKnowledgeInfoService.get(knowledgeInfoId);

        if (bizKnowledgeInfoEntity == null) {
            throw new BusinessException("知识库不存在");
        }
        String kdIds = bizKnowledgeInfoEntity.getKdIds();
        if (StringUtils.isBlank(kdIds)) {
            return result;
        }
        List<Integer> kdIdList = JsonUtils.deSerialize(kdIds, new TypeReference<List<Integer>>() {
        }.getType());
        if (CollectionUtils.isEmpty(kdIdList)) {
            return result;
        }

        UserBaseEntity currentUser = BlContext.getCurrentUserNotException();
        List<BizKnowledgeDocumentEntity> entities = knowledgeService.searchDocuments(search, trainStatus, kdIdList, currentUser.getUserId().toString(), pagingInfo);
        if (CollectionUtils.isNotEmpty(entities)) {
            result = entities.stream().map(entity -> {
                BizKnowledgeDocumentDto dto = new BizKnowledgeDocumentDto();
                BeanUtil.copyProperties(entity, dto);
                dto.setDocumentSize(FileUtils.formatFileSize(entity.getDocumentSize()));
                return dto;
            }).collect(Collectors.toList());
        }
        return result;
    }

    @Override
    public List<BizKnowledgeDocumentDto> getKdIdsByKnowledgeInfoId(Integer knowledgeInfoId) throws Exception {
        Assert.notNull(knowledgeInfoId);
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = bizKnowledgeInfoService.get(knowledgeInfoId);
        if (bizKnowledgeInfoEntity == null) {
            throw new BusinessException("知识库不存在");
        }
        List<BizKnowledgeDocumentDto> res = new ArrayList<>();
        String kdIds = bizKnowledgeInfoEntity.getKdIds();
        if (StringUtils.isNotBlank(kdIds)) {
            List<Integer> kdIdList = JsonUtils.deSerialize(kdIds, new TypeReference<List<Integer>>() {
            }.getType());
            if (CollectionUtils.isEmpty(kdIdList)) {
                return res;
            }
            List<BizKnowledgeDocumentEntity> entities = knowledgeService.getListByKdIds(kdIdList);
            if (CollectionUtils.isNotEmpty(entities)) {
                res = entities.stream().map(entity -> {
                    BizKnowledgeDocumentDto dto = new BizKnowledgeDocumentDto();
                    BeanUtil.copyProperties(entity, dto);
                    dto.setDocumentSize(FileUtils.formatFileSize(entity.getDocumentSize()));
                    return dto;
                }).collect(Collectors.toList());
            }
        }
        return res;
    }

    @Override
    public List<BizKnowledgeDocumentDto> getListByKdIds(List<Integer> kdIds) {
        Assert.notEmpty(kdIds);
        List<BizKnowledgeDocumentDto> res = new ArrayList<>();
        List<BizKnowledgeDocumentEntity> entities = knowledgeService.getListByKdIds(kdIds);
        if (CollectionUtils.isNotEmpty(entities)) {
            res = entities.stream().map(entity -> {
                BizKnowledgeDocumentDto dto = new BizKnowledgeDocumentDto();
                BeanUtil.copyProperties(entity, dto);
                dto.setDocumentSize(FileUtils.formatFileSize(entity.getDocumentSize()));
                return dto;
            }).collect(Collectors.toList());
        }
        return res;
    }

    @Override
    public BizKnowledgeInfoDto getKnowledgeDetail(Integer knowledgeInfoId) throws Exception {
        Assert.notNull(knowledgeInfoId);
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = bizKnowledgeInfoService.get(knowledgeInfoId);
        return BizKnowledgeInfoConvert.entityToDto(bizKnowledgeInfoEntity);
    }

    @Override
    public List<BizKnowledgeInfoDto> getKnowledgeList(String search, String trainStatus, PagingInfo pagingInfo) throws Exception {
        List<BizKnowledgeInfoDto> result = new ArrayList<>();
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        KnowledgeInfosQueryCondition condition = new KnowledgeInfosQueryCondition();
        condition.setMemberId(userBaseEntity.getUserId().toString());
        if (StringUtils.isNoneBlank(search)) {
            condition.setQuery(search);
        }
        if (StringUtils.isNoneBlank(trainStatus)) {
            condition.setTrainStatus(trainStatus);
        }
        List<KnowledgeInfosQueryItem> items = bizKnowledgeInfoService.knowledgeInfos(condition, pagingInfo);
        if (CollectionUtils.isNotEmpty(items)) {
            result = items.stream().map(item -> {
                BizKnowledgeInfoDto dto = new BizKnowledgeInfoDto();
                BeanUtil.copyProperties(item, dto);
                dto.setDesc(item.getKnowledgeDesc());
                return dto;
            }).collect(Collectors.toList());
        }
        return result;
    }

    @Override
    public BizKnowledgeInfoDto createKnowledge(BizKnowledgeInfoDto dto) throws Exception {
        Assert.notNull(dto.getKnowledgeName(), "知识库名称不能为空");
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = new BizKnowledgeInfoEntity();
        bizKnowledgeInfoEntity.setKnowledgeName(dto.getKnowledgeName());
        bizKnowledgeInfoEntity.setMemberId(userBaseEntity.getUserId().toString());
        bizKnowledgeInfoEntity.setTrainStatus(KnowledgeConstant.TrainStatus.UNOPENED);
        bizKnowledgeInfoEntity.setKnowledgeDesc(dto.getDesc());
        BizKnowledgeInfoEntity entity = bizKnowledgeInfoService.save(bizKnowledgeInfoEntity);
        return BizKnowledgeInfoConvert.entityToDto(entity);
    }

    @Override
    public BizKnowledgeInfoDto updateKnowledgeInfo(BizKnowledgeInfoDto dto) throws Exception {
        Assert.notNull(dto.getId());
        Assert.notBlank(dto.getKnowledgeName(), "知识库名称不能为空");
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = new BizKnowledgeInfoEntity();
        bizKnowledgeInfoEntity.setId(dto.getId());
        bizKnowledgeInfoEntity.setKnowledgeName(dto.getKnowledgeName());
        bizKnowledgeInfoEntity.setKnowledgeDesc(dto.getDesc());
        BizKnowledgeInfoEntity entity = bizKnowledgeInfoService.update(bizKnowledgeInfoEntity);
        return BizKnowledgeInfoConvert.entityToDto(entity);
    }

    @Override
    public void enableKnowledgeInfo(Integer knowledgeInfoId, String isOpen) throws Exception {
        Assert.notNull(knowledgeInfoId);
        Assert.notBlank(isOpen);
        Assert.isTrue(isOpen.equals(CommonConstant.IsDeleted.N) || isOpen.equals(CommonConstant.IsDeleted.Y), "isOpen must Y or N");
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = new BizKnowledgeInfoEntity();
        bizKnowledgeInfoEntity.setId(knowledgeInfoId);
        bizKnowledgeInfoEntity.setIsOpen(isOpen);
        bizKnowledgeInfoService.update(bizKnowledgeInfoEntity);
    }

    @Override
    public void deleteKnowledgeInfo(Integer knowledgeInfoId) throws Exception {
        Assert.notNull(knowledgeInfoId);
        bizKnowledgeInfoService.deletedById(knowledgeInfoId);
    }

    @Override
    public void openKnowledgeChunk(UpsertChunkInfoDto dto) {
        Assert.notNull(dto.getIsOpen());
        Assert.notNull(dto.getChunkRelationId());
        Assert.notNull(dto.getKdId());
        knowledgeService.openKnowledgeChunk(dto.getKdId(), dto.getChunkRelationId(), dto.getIsOpen());
    }

    @Override
    public void deleteKnowledgeChunk(UpsertChunkInfoDto dto) {
        Assert.notNull(dto.getKdId());
        Assert.notNull(dto.getChunkRelationId());
        knowledgeService.deleteKnowledgeChunk(dto.getKdId(), dto.getChunkRelationId());
    }

    @Override
    public void updateKnowledgeChunkDoc(UpsertChunkInfoDto dto) {
        Assert.notNull(dto.getChunkContent());
        Assert.notNull(dto.getChunkRelationId());
        Assert.notNull(dto.getKdId());
        knowledgeService.updateKnowledgeChunkDoc(dto.getKdId(), dto.getChunkRelationId(), dto.getChunkContent());
    }

    @Override
    public void addKnowledgeChunk(UpsertChunkInfoDto dto) {
        Assert.notNull(dto.getChunkContent());
        Assert.notNull(dto.getKdId());
        Assert.notNull(dto.getChunkSort());
        knowledgeService.addKnowledgeChunk(dto.getKdId(), dto.getChunkContent(), dto.getChunkSort());
    }

    @Override
    public GetKnowledgeChunkInfoResult getChunks(List<Integer> kdIds, String query, PagingInfo pagingInfo) {
        Assert.notEmpty(kdIds);
        return knowledgeService.getChunkInfo(kdIds, query, pagingInfo);
    }
}
