package cn.com.poc.message.service.impl;


import cn.com.poc.common.constant.CommonConstant;
import cn.com.poc.common.utils.Assert;
import cn.com.poc.common.utils.JsonUtils;
import cn.com.poc.knowledge.constant.KnowledgeConstant;
import cn.com.poc.knowledge.entity.BizKnowledgeDocumentEntity;
import cn.com.poc.knowledge.entity.BizKnowledgeInfoEntity;
import cn.com.poc.knowledge.entity.BizKnowledgeTrainLogEntity;
import cn.com.poc.knowledge.service.BizKnowledgeDocumentService;
import cn.com.poc.knowledge.service.BizKnowledgeInfoService;
import cn.com.poc.knowledge.service.BizKnowledgeTrainLogService;
import cn.com.poc.message.entity.KnowledgeTrainStatusMessage;
import cn.com.poc.message.entity.TrainKnowledgeMessage;
import cn.com.poc.message.service.KnowledgeConsumerService;
import cn.com.poc.message.service.KnowledgeProducerService;
import cn.com.poc.message.topic.KnowledgeTopic;
import cn.com.poc.thirdparty.resource.demand.ai.aggregate.DemandKnowledgeService;
import cn.com.yict.framemax.core.exception.BusinessException;
import cn.com.yict.framemax.tumbleweed.client.annotation.Consumer;
import com.alibaba.fastjson.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.parameters.P;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;


@Service
public class KnowledgeConsumerServiceImpl implements KnowledgeConsumerService {

    final private Logger logger = LoggerFactory.getLogger(KnowledgeConsumerService.class);

    @Resource
    private BizKnowledgeTrainLogService bizKnowledgeTrainLogService;

    @Resource
    private BizKnowledgeDocumentService bizKnowledgeDocumentService;

    @Resource
    private DemandKnowledgeService demandKnowledgeService;

    @Resource
    private KnowledgeProducerService knowledgeProducerService;

    @Resource
    private BizKnowledgeInfoService bizKnowledgeInfoService;


    /**
     * 训练知识库
     *
     * @param message
     * @return
     */
    @Override
    @Consumer(topic = KnowledgeTopic.TRAIN_KNOWLEDGE, scale = 3, retry = true)
    public void trainKnowledge(TrainKnowledgeMessage message) throws Exception {
        //修改训练状态
        KnowledgeTrainStatusMessage trainStatusMessage = new KnowledgeTrainStatusMessage();
        trainStatusMessage.setStatus(KnowledgeConstant.TrainStatus.TRAINING);
        trainStatusMessage.setKdId(message.getKid());
        trainStatusMessage.setKnowledgeInfoId(message.getKnowledgeInfoId());
        knowledgeProducerService.trainStatusUpdate(trainStatusMessage);
        try {
            String knowledgeId = demandKnowledgeService.trainKnowledge(message.getFileUrl(), message.getSegmentationConfig());

            //训练日志
            BizKnowledgeTrainLogEntity bizKnowledgeTrainLogEntity = new BizKnowledgeTrainLogEntity();
            bizKnowledgeTrainLogEntity.setKdId(message.getKid());
            bizKnowledgeTrainLogEntity.setTimestamp(System.currentTimeMillis());
            bizKnowledgeTrainLogEntity.setTrainStatus(KnowledgeConstant.TrainStatus.COMPLETE);
            bizKnowledgeTrainLogEntity.setFailureLog("");
            bizKnowledgeTrainLogEntity.setIsDeleted(CommonConstant.IsDeleted.N);
            bizKnowledgeTrainLogService.save(bizKnowledgeTrainLogEntity);

            //训练完成
            KnowledgeTrainStatusMessage completeMessage = new KnowledgeTrainStatusMessage();
            completeMessage.setStatus(KnowledgeConstant.TrainStatus.COMPLETE);
            completeMessage.setKdId(message.getKid());
            completeMessage.setKnowledgeId(knowledgeId);
            completeMessage.setKnowledgeInfoId(message.getKnowledgeInfoId());
            knowledgeProducerService.trainStatusUpdate(completeMessage);
        } catch (BusinessException e) {
            logger.warn("--------------message:{},知识库训练失败：{}------------", message, e.getMessage());

            //记录状态 训练失败
            trainStatusMessage.setStatus(KnowledgeConstant.TrainStatus.FAIL);
            trainStatusMessage.setKdId(message.getKid());
            knowledgeProducerService.trainStatusUpdate(trainStatusMessage);

            logger.warn("-------保存知识库训练失败状态----------");
            //训练日志
            BizKnowledgeTrainLogEntity bizKnowledgeTrainLogEntity = new BizKnowledgeTrainLogEntity();
            bizKnowledgeTrainLogEntity.setKdId(message.getKid());
            bizKnowledgeTrainLogEntity.setTimestamp(System.currentTimeMillis());
            bizKnowledgeTrainLogEntity.setTrainStatus(KnowledgeConstant.TrainStatus.FAIL);
            bizKnowledgeTrainLogEntity.setFailureLog(e.getMessage());
            bizKnowledgeTrainLogEntity.setIsDeleted(CommonConstant.IsDeleted.N);
            bizKnowledgeTrainLogService.save(bizKnowledgeTrainLogEntity);
            logger.warn("-------保存知识库训练失败日志----------");


            BizKnowledgeInfoEntity bizKnowledgeInfoEntity = bizKnowledgeInfoService.get(message.getKnowledgeInfoId());
            bizKnowledgeInfoEntity.setTrainStatus(KnowledgeConstant.TrainStatus.FAIL);
            bizKnowledgeInfoService.update(bizKnowledgeInfoEntity);
            logger.warn("-------保存知识库训练失败状态----------");

        }
    }


    @Override
    @Consumer(topic = KnowledgeTopic.TRAIN_STATUS, scale = 1, retry = true)
    public void trainStatusUpdate(KnowledgeTrainStatusMessage message) throws Exception {
        BizKnowledgeDocumentEntity bizKnowledgeDocumentEntity = new BizKnowledgeDocumentEntity();
        bizKnowledgeDocumentEntity.setTrainStatus(message.getStatus());
        bizKnowledgeDocumentEntity.setKnowledgeId(message.getKnowledgeId());
        bizKnowledgeDocumentService.update(message.getKdId(), bizKnowledgeDocumentEntity);
        knowledgeProducerService.knowledgeInfoStatusCheck(message);
    }


    @Override
    @Consumer(topic = KnowledgeTopic.KNOWLEDGE_INFO_STATUS_CHECK, scale = 3, retry = true)
    public void knowledgeInfoStatusCheck(KnowledgeTrainStatusMessage knowledgeTrainStatusMessage) throws Exception {
        Assert.notNull(knowledgeTrainStatusMessage.getKnowledgeInfoId());
        BizKnowledgeInfoEntity bizKnowledgeInfoEntity = bizKnowledgeInfoService.get(knowledgeTrainStatusMessage.getKnowledgeInfoId());
        if (!KnowledgeConstant.TrainStatus.TRAINING.equals(bizKnowledgeInfoEntity.getTrainStatus())) {
            return;
        }

        String kdIds = bizKnowledgeInfoEntity.getKdIds();
        List<Integer> kdIdList = JsonUtils.deSerialize(kdIds, new TypeReference<List<Integer>>() {
        }.getType());

        boolean isAllComplete = true;
        logger.info("-------知识库训练状态检查，kdIds:{}-------", kdIds);
        for (Integer kdId : kdIdList) {
            BizKnowledgeDocumentEntity documentEntity = bizKnowledgeDocumentService.get(kdId);
            if (!documentEntity.getTrainStatus().equals(KnowledgeConstant.TrainStatus.COMPLETE)) {
                isAllComplete = false;
                break;
            }
        }

        if (isAllComplete) {
            logger.info("-------知识库训练状态检查，全部完成, knowledgeInfoId:{}-------", knowledgeTrainStatusMessage.getKnowledgeInfoId());
            bizKnowledgeInfoEntity.setTrainStatus(KnowledgeConstant.TrainStatus.COMPLETE);
            bizKnowledgeInfoService.update(bizKnowledgeInfoEntity);
        }

    }
}
