package cn.com.poc.agent_application.rest.impl;

import cn.com.poc.agent_application.aggregate.AgentApplicationInfoService;
import cn.com.poc.agent_application.convert.AgentApplicationInfoConvert;
import cn.com.poc.agent_application.dto.*;
import cn.com.poc.agent_application.entity.*;
import cn.com.poc.agent_application.query.AgentApplicationInfoQueryCondition;
import cn.com.poc.agent_application.rest.AgentApplicationInfoRest;
import cn.com.poc.agent_application.service.*;
import cn.com.poc.common.annotation.RedisLimit;
import cn.com.poc.common.constant.CommonConstant;
import cn.com.poc.common.service.RedisService;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.common.utils.JsonUtils;
import cn.com.poc.knowledge.aggregate.KnowledgeService;
import cn.com.poc.support.security.oauth.entity.UserBaseEntity;
import cn.com.poc.thirdparty.resource.demand.ai.entity.dialogue.Tool;
import cn.com.poc.thirdparty.resource.demand.ai.function.LargeModelFunctionEnum;
import cn.com.poc.thirdparty.resource.demand.ai.function.long_memory.SetLongMemoryConstants;
import cn.com.poc.thirdparty.resource.demand.ai.function.value_memory.GetValueMemory;
import cn.com.poc.thirdparty.resource.demand.ai.function.value_memory.SetValueMemoryConstants;
import cn.com.yict.framemax.core.i18n.I18nMessageException;
import cn.com.yict.framemax.data.model.PagingInfo;
import cn.hutool.core.collection.ListUtil;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.*;
import java.util.stream.Collectors;

@Component
public class AgentApplicationInfoRestImpl implements AgentApplicationInfoRest {

    @Resource
    private BizAgentApplicationInfoService bizAgentApplicationInfoService;

    @Resource
    private AgentApplicationInfoService agentApplicationInfoService;

    @Resource
    private BizAgentApplicationPublishService bizAgentApplicationPublishService;

    @Resource
    private BizAgentApplicationLargeModelListService bizAgentApplicationLargeModelListService;

    @Resource
    private KnowledgeService knowledgeService;

    @Resource
    private RedisService redisService;

    @Resource
    private BizMemberAgentApplicationCollectService bizMemberAgentApplicationCollectService;

    @Resource
    private BizAgentApplicationMallService bizAgentApplicationMallService;

    @Override
    public AgentApplicationInfoDto saveVariableStructure(String agentId, List<Variable> variableStructure) {
        Assert.notNull(agentId);
        BizAgentApplicationInfoEntity bizAgentApplicationInfoEntity = bizAgentApplicationInfoService.saveVariableStructure(agentId, variableStructure);
        if (bizAgentApplicationInfoEntity == null) {
            throw new I18nMessageException("exception/application.does.not.exist");
        }
        return AgentApplicationInfoConvert.entityToDto(bizAgentApplicationInfoEntity);
    }

    @Override
    public List<AgentApplicationInfoDto> getListByMember(AgentApplicationInfoSearchDto dto, PagingInfo pagingInfo) throws Exception {
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        Long userId = userBaseEntity.getUserId();
        AgentApplicationInfoQueryCondition condition = new AgentApplicationInfoQueryCondition();
        condition.setMemberId(userId);
        if (!StringUtils.isEmpty(dto.getQuery())) {
            condition.setQuery(dto.getQuery());
        }
        if (StringUtils.isNotBlank(dto.getPublishStatus())) {
            condition.setAgentPublishStatus(dto.getPublishStatus());
        }
        List<BizAgentApplicationInfoEntity> bizAgentApplicationInfoEntities = bizAgentApplicationInfoService.agentApplicationInfoQuery(condition, pagingInfo);
        List<AgentApplicationInfoDto> resultList = new ArrayList<>();
        for (BizAgentApplicationInfoEntity entity : bizAgentApplicationInfoEntities) {
            AgentApplicationInfoDto infoDto = AgentApplicationInfoConvert.entityToDto(entity);
            // 先判断有没有收藏的查询条件
            if (StringUtils.isNotBlank(dto.getIsCollect())) {
                // 如果有查询条件，则判断符不符合条件
                if (infoDto.getBaseInfo() != null) {
                    BizMemberAgentApplicationCollectEntity collect = bizMemberAgentApplicationCollectService.getByAgentId(infoDto.getBaseInfo().getAgentId());
                    // 如果符合条件，则把值加入
                    if (collect != null && collect.getIsCollect().equals(dto.getIsCollect())) {
                        infoDto.setIsCollect(collect.getIsCollect());
                    } else if (collect == null && CommonConstant.IsDeleted.N.equals(dto.getIsCollect())) {
                        infoDto.setIsCollect(CommonConstant.IsDeleted.N);
                    } else {
                        // 如果不符合条件，则continue
                        continue;
                    }
                } else {
                    continue;
                }
            } else {
                // 如何没有查询条件，则直接按照原来的逻辑走
                // 设置当前用户是否收藏了
                if (infoDto.getBaseInfo() != null) {
                    BizMemberAgentApplicationCollectEntity collect = bizMemberAgentApplicationCollectService.getByAgentId(infoDto.getBaseInfo().getAgentId());
                    if (collect != null) {
                        infoDto.setIsCollect(collect.getIsCollect());
                    } else {
                        infoDto.setIsCollect(CommonConstant.IsDeleted.N);
                    }
                }
            }
            // 判断是否发布了，若发布则设置agentPublishId
            BizAgentApplicationPublishEntity publishEntity = bizAgentApplicationPublishService.getByAgentId(entity.getAgentId());
            if (publishEntity != null) {
                infoDto.setAgentPublishId(publishEntity.getId());
                // 设置当前作品是否上架
                BizAgentApplicationMallEntity mallEntity = bizAgentApplicationMallService.getByAgentPublishId(publishEntity.getId());
                if (mallEntity != null) {
                    infoDto.setIsSale(mallEntity.getIsSale());
                } else {
                    infoDto.setIsSale(CommonConstant.IsDeleted.N);
                }
            } else {
                infoDto.setAgentPublishId(null);
            }
            resultList.add(infoDto);
        }
        return resultList;

    }

    @Override
    public AgentApplicationInfoDto saveOrUpdate(AgentApplicationInfoDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getBaseInfo());
        Assert.notNull(dto.getCommConfig());
        Assert.notNull(dto.getCommModelConfig());
        Assert.notNull(dto.getKnowledgeConfig());

        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        if (userBaseEntity == null) {
            throw new I18nMessageException("exception/user.does.not.exist");
        }
        Long userId = userBaseEntity.getUserId();
        dto.getBaseInfo().setMemberId(userId.intValue());
        BizAgentApplicationInfoEntity entity = AgentApplicationInfoConvert.dtoToEntity(dto);

        return AgentApplicationInfoConvert.entityToDto(agentApplicationInfoService.saveOrUpdate(entity));
    }


    public AgentApplicationInfoDto update(AgentApplicationInfoDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getBaseInfo());
        Assert.notNull(dto.getBaseInfo().getAgentId());
        Assert.notNull(dto.getCommConfig());
        Assert.notNull(dto.getCommModelConfig());
        Assert.notNull(dto.getKnowledgeConfig());
        BizAgentApplicationInfoEntity entity = AgentApplicationInfoConvert.dtoToEntity(dto);
        return AgentApplicationInfoConvert.entityToDto(bizAgentApplicationInfoService.update(entity));
    }

    @Override
    public AgentApplicationInfoDto getInfo(String agentId) throws Exception {
        Assert.notNull(agentId);
        BizAgentApplicationInfoEntity entity = bizAgentApplicationInfoService.getByAgentId(agentId);
        if (entity == null) {
            throw new I18nMessageException("exception/application.does.not.exist");
        }
        return AgentApplicationInfoConvert.entityToDto(entity);
    }

    @Override
    public void delete(String agentId) throws Exception {
        Assert.notNull(agentId);
        agentApplicationInfoService.deletedAgentApplication(agentId);
    }

    @Override
    public boolean updateAndPublish(AgentApplicationInfoDto dto) throws Exception {
        Assert.notNull(dto);
        Assert.notNull(dto.getBaseInfo());
        Assert.notNull(dto.getBaseInfo().getAgentId());
        Assert.notNull(dto.getBaseInfo().getAgentTitle());
        Assert.notNull(dto.getCommConfig());
        Assert.notNull(dto.getCommModelConfig());
        Assert.notNull(dto.getKnowledgeConfig());
        BizAgentApplicationInfoEntity bizAgentApplicationInfoEntity = AgentApplicationInfoConvert.dtoToEntity(dto);
        agentApplicationInfoService.updateAndPublish(bizAgentApplicationInfoEntity);
        return true;
    }

    @Override
    public void preview(AgentApplicationPreviewDto dto, HttpServletResponse httpServletResponse) throws IOException {
        Assert.notNull(dto);
        Assert.notNull(dto.getAgentId());
        Assert.notEmpty(dto.getMessages());
        try {
            String agentId = dto.getAgentId();
            BizAgentApplicationInfoEntity infoEntity = bizAgentApplicationInfoService.getByAgentId(agentId);
            if (infoEntity == null) {
                throw new I18nMessageException("exception/application.does.not.exist");
            }
            //获取知识库配置
            List<Integer> kdIds = knowledgeService.getKdIdsByKnowledgeInfoIds(infoEntity.getKnowledgeIds());

            //配置对话function
            List<Tool> tools = new ArrayList<>();
            //开启对话变量
            if (CollectionUtils.isNotEmpty(infoEntity.getVariableStructure())) {
                String functionName = LargeModelFunctionEnum.set_value_memory.name();
                String llmConfig = LargeModelFunctionEnum.valueOf(functionName).getFunction().getVariableStructureLLMConfig(infoEntity.getVariableStructure()).get(0);
                Tool tool = JsonUtils.deSerialize(llmConfig, Tool.class);
                tools.add(tool);

                //初始化变量函数
                Map<Object, Object> map = GetValueMemory.get(agentId);
                List<Variable> variableStructure = infoEntity.getVariableStructure();
                if (MapUtils.isEmpty(map)) {
                    for (Variable variable : variableStructure) {
                        String key = variable.getKey();
                        String variableDefault = variable.getVariableDefault();
                        JSONObject jsonObject = new JSONObject();
                        jsonObject.put("contentName", key);
                        jsonObject.put("contentValue", variableDefault);
                        LargeModelFunctionEnum.valueOf(functionName).getFunction().doFunction(jsonObject.toJSONString(), agentId);
                    }
                }

            }
            //开启长期记忆
            if (CommonConstant.YOrN.Y.equals(infoEntity.getIsLongMemory())) {
                String functionName = LargeModelFunctionEnum.set_long_memory.name();
                String llmConfig = LargeModelFunctionEnum.valueOf(functionName).getFunction().getLLMConfig().get(0);
                Tool tool = JsonUtils.deSerialize(llmConfig, Tool.class);
                tools.add(tool);
            }

            //对话大模型配置
            String model = StringUtils.isNotBlank(dto.getModelNickName()) ? dto.getModelNickName() : infoEntity.getLargeModel();
            Float topP = dto.getTopP() == null ? infoEntity.getTopP() : dto.getTopP();
            Float temperature = dto.getTemperature() == null ? infoEntity.getTemperature() : dto.getTemperature();
            String agentSystem = StringUtils.isBlank(dto.getAgentSystem()) ? infoEntity.getAgentSystem() : dto.getAgentSystem();

            //调用应用服务
            agentApplicationInfoService.callAgentApplication(agentId, model, infoEntity.getUnitIds()
                    , agentSystem, kdIds.toArray(new Integer[0]), infoEntity.getCommunicationTurn(), topP, temperature
                    , dto.getMessages(), tools, httpServletResponse);
        } catch (
                Exception e) {
            httpServletResponse.setContentType("text/event-stream");
            PrintWriter writer = httpServletResponse.getWriter();
            writer.write("data: {\"code\":-1,\"message\":\"" + e.getLocalizedMessage() + "\"} \n\n");
            writer.write("data: [DONE]\n\n");
            writer.flush();
        }
    }

    @Override
    public List<BizAgentApplicationLargeModelListDto> getLargeModelList() throws Exception {
        List<BizAgentApplicationLargeModelListEntity> entities = bizAgentApplicationLargeModelListService.findByExample(new BizAgentApplicationLargeModelListEntity(), null);
        Map<String, List<BizAgentApplicationLargeModelListEntity>> map = entities.stream()
                .collect(Collectors.groupingBy(BizAgentApplicationLargeModelListEntity::getOwner));

        List<BizAgentApplicationLargeModelListDto> result = new ArrayList<>();
        for (String owner : map.keySet()) {
            BizAgentApplicationLargeModelListDto dto = new BizAgentApplicationLargeModelListDto();
            List<BizAgentApplicationLargeModelListEntity> modelListEntities = map.get(owner);
            dto.setOwner(owner);
            dto.setModels(modelListEntities.stream().map(BizAgentApplicationLargeModelListEntity::getModelNickName).collect(Collectors.toList()));
            dto.setIcon(modelListEntities.get(0).getIconUrl());
            result.add(dto);
        }
        return result;
    }

    @Override
    public BizAgentApplicationLargeModelListDto getLargeModelInfo(String query) throws Exception {
        Assert.notNull(query);
        BizAgentApplicationLargeModelListEntity bizAgentApplicationLargeModelListEntity = new BizAgentApplicationLargeModelListEntity();
        bizAgentApplicationLargeModelListEntity.setModelNickName(query);
        List<BizAgentApplicationLargeModelListEntity> entities = bizAgentApplicationLargeModelListService.findByExample(bizAgentApplicationLargeModelListEntity, null);
        if (CollectionUtils.isEmpty(entities)) {
            throw new I18nMessageException("exception/model.not.exist");
        }
        BizAgentApplicationLargeModelListEntity entity = entities.get(0);
        BizAgentApplicationLargeModelListDto result = new BizAgentApplicationLargeModelListDto();
        result.setModels(ListUtil.toList(entity.getModelNickName()));
        result.setOwner(entity.getOwner());
        result.setIcon(entity.getIconUrl());
        return result;
    }

    @Override
    public void createAgentSystem(AgentApplicationGCDto dto, HttpServletResponse response) throws Exception {
        Assert.notNull(dto.getInput());
        agentApplicationInfoService.createAgentSystem(dto.getInput(), response);
    }

    @Override
    public List<String> createFeaturedQuestions(AgentApplicationGCDto dto) {
        return agentApplicationInfoService.createFeaturedQuestions(dto.getAgentTitle(), dto.getAgentDesc());
    }

    @Override
    public String createPreamble(AgentApplicationGCDto dto) {
        return agentApplicationInfoService.createPreamble(dto.getAgentTitle(), dto.getAgentDesc(), dto.getAgentSystem());
    }

    @Override
    public String createAgentApplicationAvatar(AgentApplicationGCDto dto) throws Exception {
        Assert.isTrue(StringUtils.isNotBlank(dto.getAgentDesc()) || StringUtils.isNotBlank(dto.getAgentTitle()));
        return agentApplicationInfoService.createAgentIcon(dto.getAgentTitle(), dto.getAgentDesc());
    }

    @Override
    public AgentApplicationGCDto createAgentTitleAndDesc(AgentApplicationGCDto dto) {
        Assert.notNull(dto.getInput());
        CreateAgentTitleAndDescEntity entity = agentApplicationInfoService.createAgentTitleAndDesc(dto.getInput());
        if (null == entity) {
            throw new I18nMessageException("exception/generation.failed.please.try.again.later");
        }
        AgentApplicationGCDto result = new AgentApplicationGCDto();
        result.setAgentTitle(entity.getAgentTitle());
        result.setAgentDesc(entity.getAgentDesc());
        return result;
    }

    @Override
    @RedisLimit(key = "collect:agent_person", currentUser = true, count = 1, timeout = 1, exceptionInfo = "exception/too.many.requests")
    public void collectOrCancelAgentInPerson(String agentId) throws Exception {
        Assert.notNull(agentId);
        agentApplicationInfoService.collectOrCancelAgentInPerson(agentId);
    }

    @Override
    public List<AgentApplicationValueMemoryDto> getVariableList(String agentId) {
        Map<Object, Object> map = GetValueMemory.get(agentId);
        List<AgentApplicationValueMemoryDto> result = new ArrayList<>();
        if (MapUtils.isEmpty(map)) {
            BizAgentApplicationInfoEntity infoEntity = bizAgentApplicationInfoService.getByAgentId(agentId);
            List<Variable> variableStructure = infoEntity.getVariableStructure();
            if (CollectionUtils.isEmpty(variableStructure)) {
                return result;
            }
            for (Variable variable : variableStructure) {
                AgentApplicationValueMemoryDto valueMemoryDto = new AgentApplicationValueMemoryDto();
                valueMemoryDto.setKey(variable.getKey());
                valueMemoryDto.setValue(variable.getVariableDefault());
                result.add(valueMemoryDto);
            }
        } else {
            Set<Object> keySet = map.keySet();
            for (Object key : keySet) {
                AgentApplicationValueMemoryDto valueMemoryDto = new AgentApplicationValueMemoryDto();
                valueMemoryDto.setKey(key.toString());
                valueMemoryDto.setValue(map.get(key).toString());
                result.add(valueMemoryDto);
            }
        }
        return result;
    }

    @Override
    public List<AgentLongMemoryDto> getLongMemoryList(String agentId) {
        Assert.notNull(agentId);
        List<AgentLongMemoryDto> result = new ArrayList<>();
        String contentKey = SetLongMemoryConstants.REDIS_PREFIX + agentId + ":" + BlContext.getCurrentUserNotException().getUserId().toString();
        Map<Object, Object> map = redisService.hmget(contentKey);
        Set<Object> keySet = map.keySet();

        for (Object mapKey : keySet) {
            AgentLongMemoryDto entity = new AgentLongMemoryDto();
            entity.setContent(map.get(mapKey).toString());
            entity.setTimestamp(mapKey.toString());
            result.add(entity);
        }
        return result.stream().sorted(Comparator.comparing(AgentLongMemoryDto::getTimestamp).reversed()).collect(Collectors.toList());
    }

    @Override
    public void deleteLongMemoryByKey(String agentId, String timestamp) {
        Assert.notNull(agentId);
        Assert.notNull(timestamp);
        String contentKey = SetLongMemoryConstants.REDIS_PREFIX + agentId + ":" + BlContext.getCurrentUserNotException().getUserId().toString();
        redisService.hdel(contentKey, timestamp);
    }

    @Override
    public void deleteLongMemory(String agentId) {
        Assert.notNull(agentId);
        String contentKey = SetLongMemoryConstants.REDIS_PREFIX + agentId + ":" + BlContext.getCurrentUserNotException().getUserId().toString();
        redisService.del(contentKey);
    }
}