package cn.com.poc.equity.aggregate;

import cn.com.poc.equity.constants.EquityEnum;
import cn.com.poc.equity.constants.ModifyEventEnum;
import cn.com.poc.equity.domain.MemberEquityInfo;
import cn.com.poc.equity.domain.modifyEquityInfo.ModifyEventInfo;
import cn.com.poc.equity.entity.BizMemberEquityEntity;

/**
 * 用户权益服务
 *
 * @author alex.yao
 * @date 2024/12/6
 */
public interface MemberEquityService {

    /**
     * 新用户权益初始化
     *
     * @param memberId 用户id
     */
    void initMemberEquity(Long memberId) throws Exception;


    /**
     * 新增积分
     *
     * @param memberId        用户id
     * @param pointNum        积分数
     * @param validityUnit    积分过期时间
     * @param modifyEventEnum 修改事件枚举 - 礼品 / 充值
     * @param modifyEventInfo 修改事件信息详情
     * @return 返回扣减积分流水号
     */
    String addPoint(Long memberId, Long pointNum, EquityEnum.VALIDITY_UNIT validityUnit, ModifyEventEnum modifyEventEnum, ModifyEventInfo modifyEventInfo) throws Exception;


    /**
     * 扣减积分
     *
     * @param memberId        用户id
     * @param pointNum        积分数
     * @param modifyEventEnum 修改事件枚举 - 使用 / 过期
     * @param modifyEventInfo 修改事件信息详情
     * @return 返回扣减积分流水号
     */
    String reducePoint(Long memberId, Long pointNum, ModifyEventEnum modifyEventEnum, ModifyEventInfo modifyEventInfo) throws Exception;

    /**
     * 积分回滚
     * 若是充值，则删除积分数据
     * 若是消费，则恢复积分数据
     *
     * @param transactionSn 需要回滚的积分流水号
     */
    void rollbackPoint(String transactionSn) throws Exception;


    /**
     * 用户权益等级变更
     * 注意：如果权益变更事件为top_up, 需要调用addPoint方法最加积分发放
     *
     * @param memberId        用户id
     * @param modifyEventEnum 变更事件
     * @param type            权益等级类型
     * @param validityUnit    权益等级有效期
     * @param modifyEventInfo 修改事件信息详情
     * @return 返回权益等级变更流水号
     */
    String changeEquityLevel(Long memberId, ModifyEventEnum modifyEventEnum, EquityEnum.TYPE type, EquityEnum.VALIDITY_UNIT validityUnit, ModifyEventInfo modifyEventInfo) throws Exception;


    /**
     * 获取用户剩余可用积分
     *
     * @param memberId
     * @return
     */
    Long getMemberPoints(Long memberId);

    /**
     * 获取用户权益信息
     *
     * @param memberId
     * @return
     */
    MemberEquityInfo getEquityInfo(Long memberId) throws Exception;
}
