package cn.com.poc.data_analyze.constants;

import cn.com.poc.common.utils.Assert;
import cn.com.poc.common.utils.DateUtils;
import cn.com.poc.data_analyze.domain.DataAnalyzeTimeRange;
import cn.com.poc.data_analyze.domain.TimeRange;
import cn.com.yict.framemax.core.i18n.I18nMessageException;

import java.util.Date;

/**
 * 应用数据分析-时间范围枚举
 *
 * @author alex.yao
 * @date 2024/12/11
 */
public enum DataAnalyzeTimeRangeEnum {
    customize("customize"),
    today("today"),
    week("week"),
    month("month"),
    ;

    private String type;

    DataAnalyzeTimeRangeEnum(String type) {
        this.type = type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getType() {
        return type;
    }

    public static DataAnalyzeTimeRangeEnum getByType(String type) {
        for (DataAnalyzeTimeRangeEnum item : values()) {
            if (item.getType().equals(type)) {
                return item;
            }
        }
        throw new I18nMessageException("data-analyze/not.support.rang.type");
    }

    public static TimeRange calculateTimeRange(DataAnalyzeTimeRange dataAnalyzeTimeRange) {
        String startTime;
        String endTime;
        Date startDate;
        Date endDate;
        String rangType = dataAnalyzeTimeRange.getRangType();
        switch (DataAnalyzeTimeRangeEnum.getByType(rangType)) {
            case today:
                endDate = DateUtils.getTodayEnd();
                startDate = DateUtils.getDayStartTime(endDate);
                startTime = DateUtils.formatDate(startDate, DateUtils.yyyy_MM_dd);
                endTime = DateUtils.formatDate(endDate, DateUtils.yyyy_MM_dd_HH_mm_ss);
                break;
            case week:
                endDate = DateUtils.getTodayEnd();
                startDate = DateUtils.getDayStartTime(DateUtils.addDays(endDate, -6));
                startTime = DateUtils.formatDate(startDate, DateUtils.yyyy_MM_dd);
                endTime = DateUtils.formatDate(endDate, DateUtils.yyyy_MM_dd_HH_mm_ss);
                break;
            case month:
                endDate = DateUtils.getTodayEnd();
                startDate = DateUtils.getDayStartTime(DateUtils.addMonth(endDate, -1));
                startTime = DateUtils.formatDate(startDate, DateUtils.yyyy_MM_dd);
                endTime = DateUtils.formatDate(endDate, DateUtils.yyyy_MM_dd_HH_mm_ss);
                break;
            case customize:
                Assert.notBlank(dataAnalyzeTimeRange.getStartTime());
                Assert.notBlank(dataAnalyzeTimeRange.getEndTime());
                startTime = dataAnalyzeTimeRange.getStartTime();
                endDate = DateUtils.getDayEnd(DateUtils.stringToDateShort(dataAnalyzeTimeRange.getEndTime()));
                endTime = DateUtils.formatDate(endDate, DateUtils.yyyy_MM_dd_HH_mm_ss);
                break;
            default:
                throw new I18nMessageException("data-analyze/not.support.rang.type");
        }
        TimeRange timeRange = new TimeRange();
        timeRange.setStartTime(startTime);
        timeRange.setEndTime(endTime);
        return timeRange;
    }
}
