package cn.com.poc.human_resources.rest.impl;

import cn.com.poc.common.utils.Assert;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.human_resources.aggregate.HumanResourceService;
import cn.com.poc.human_resources.convert.BizHumanResourcePositionConvert;
import cn.com.poc.human_resources.convert.BizHumanResourceResumeConvert;
import cn.com.poc.human_resources.dto.*;
import cn.com.poc.human_resources.entity.BizHumanResourcePositionEntity;
import cn.com.poc.human_resources.entity.HumanResourcePositionBaseInfo;
import cn.com.poc.human_resources.rest.HumanResourceRest;
import cn.com.yict.framemax.core.exception.BusinessException;
import cn.com.yict.framemax.data.model.PagingInfo;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author alex.yao
 * @date 2025/9/28
 */
@Component
public class HumanResourceRestImpl implements HumanResourceRest {

    @Resource
    private HumanResourceService humanResourceService;

    @Override
    public Object uploadResume(UploadResumeDto uploadResumeDto) throws Exception {
        Assert.notBlank(uploadResumeDto.getFileURL(), "文件不能为空");
        Assert.notNull(uploadResumeDto.getPositionId(), "岗位ID不能为空");

        return humanResourceService.uploadResume(
                uploadResumeDto.getFileURL(),
                uploadResumeDto.getPositionId(),
                BlContext.getCurrentUser().getUserId()
        );
    }

    @Override
    public void positionAssistant(PositionAssistantDto positionAssistantDto) throws Exception {
        Assert.notBlank(positionAssistantDto.getType(), "类型不能为空");
        humanResourceService.positionAssistant(
                positionAssistantDto.getDemand(),
                positionAssistantDto.getType(),
                positionAssistantDto.getFileInfos(),
                positionAssistantDto.getJobInformation(),
                positionAssistantDto.getQuestion(),
                positionAssistantDto.getDialoguesId(),
                BlContext.getCurrentUser().getUserId()
        );
    }

    @Override
    public void savePosition(HumanResourcePositionDto humanResourcePositionDto) throws Exception {
        Assert.notNull(humanResourcePositionDto.getBaseInfo(), "基础信息不能为空");
        Assert.notNull(humanResourcePositionDto.getJob(), "职位详情不能为空");
        //基础信息
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getJobTitle(), "职位名称不能为空");
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getDepartment(), "所属部门不能为空");
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getWorkLocation(), "工作地址不能为空");
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getJobNature(), "工作性质不能为空");
        Assert.notNull(humanResourcePositionDto.getBaseInfo().getNumberOfRecruits(), "招聘人数不能为空");
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getWorkExperience(), "工作经验不能为空");
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getEducationalRe(), "学历要求不能为空");
        Assert.notBlank(humanResourcePositionDto.getBaseInfo().getSalaryRange(), "薪资要求不能为空");
        //职位详情
        Assert.notBlank(humanResourcePositionDto.getJob().getJobDuties(), "职位描述不能为空");
        Assert.notBlank(humanResourcePositionDto.getJob().getJobRequirements(), "任职资格不能为空");

        humanResourceService.saveOrUpdatePosition(
                humanResourcePositionDto.getBaseInfo(),
                humanResourcePositionDto.getJob(),
                humanResourcePositionDto.getPor(),
                humanResourcePositionDto.getId(),
                BlContext.getCurrentUser().getUserId()
        );
    }


    @Override
    public List<HumanResourcePositionBaseInfoDto> listPositions(PagingInfo pagingInfo) throws Exception {
        List<HumanResourcePositionBaseInfo> humanResourcePositionBaseInfos = humanResourceService.listPositions(BlContext.getCurrentUser().getUserId(), pagingInfo);
        return humanResourcePositionBaseInfos.stream()
                .map(info -> {
                    HumanResourcePositionBaseInfoDto dto = new HumanResourcePositionBaseInfoDto();
                    dto.setId(info.getId());
                    dto.setJobTitle(info.getJobTitle());
                    dto.setWorkLocation(info.getWorkLocation());
                    dto.setJobNature(info.getJobNature());
                    dto.setWorkExperience(info.getWorkExperience());
                    dto.setResumeCount(info.getResumeCount());
                    return dto;
                })
                .collect(Collectors.toList());
    }

    @Override
    public HumanResourcePositionDto getPositionDetail(Long positionId) throws Exception {
        BizHumanResourcePositionEntity positionEntity = humanResourceService.getPositionDetail(BlContext.getCurrentUser().getUserId(), positionId);
        if (positionEntity == null) {
            throw new BusinessException("该岗位信息不存在");
        }
        HumanResourcePositionDto resourcePositionDto = BizHumanResourcePositionConvert.entityToHumanResourcePositionDto(positionEntity);
        resourcePositionDto.setId(positionId);
        return resourcePositionDto;
    }

    @Override
    public void deletePositions(Long... positionIds) throws Exception {
        Assert.notEmpty(positionIds, "请选择要删除的岗位");
        humanResourceService.deletePositions(BlContext.getCurrentUser().getUserId(), positionIds);
    }

    @Override
    public void saveResume(SaveResumeDto saveResumeDto) throws Exception {
        Assert.notBlank(saveResumeDto.getFileURL(), "文件不能为空");
        Assert.notNull(saveResumeDto.getPositionId(), "岗位id不能为空");
        Assert.notNull(saveResumeDto.getResume());
        Assert.notNull(saveResumeDto.getResume().getInfo(), "简历基础信息不能为空");
        Assert.notNull(saveResumeDto.getResume().getTag(), "简历标签信息不能为空");
        if (saveResumeDto.getId() != null) {
            humanResourceService.updateResume(saveResumeDto.getId(), BlContext.getCurrentUser().getUserId(), saveResumeDto.getFileURL(), saveResumeDto.getPositionId(), saveResumeDto.getResume());
        } else {
            humanResourceService.saveResume(BlContext.getCurrentUser().getUserId(), saveResumeDto.getFileURL(), saveResumeDto.getPositionId(), saveResumeDto.getResume());
        }
    }

    @Override
    public List<ResumeDto> listResumes(Long positionId, PagingInfo pagingInfo) throws Exception {
        return humanResourceService.listResumes(BlContext.getCurrentUser().getUserId(), positionId, pagingInfo)
                .stream()
                .map(BizHumanResourceResumeConvert::entityToResumeDto)
                .collect(Collectors.toList());
    }

    @Override
    public ResumeDto getResumeDetail(Long resumeId) throws Exception {
        return BizHumanResourceResumeConvert.entityToResumeDto(humanResourceService.getResumeDetail(BlContext.getCurrentUser().getUserId(), resumeId));
    }

    @Override
    public void deleteResumes(Long... resumeIds) throws Exception {
        Assert.notEmpty(resumeIds, "请选择要删除的简历");
        humanResourceService.deleteResumes(BlContext.getCurrentUser().getUserId(), resumeIds);
    }
}
