package cn.com.poc.user.rest.impl;

import cn.com.poc.common.utils.Assert;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.support.security.oauth.constants.OauthConstants;
import cn.com.poc.support.security.oauth.entity.UserBaseEntity;
import cn.com.poc.user.aggregation.MemberInfoService;
import cn.com.poc.user.builder.LoginChannelBuilder;
import cn.com.poc.user.convert.MemberInfoConvert;
import cn.com.poc.user.dto.MemberInfoDto;
import cn.com.poc.user.dto.MemberLoginRequestDto;
import cn.com.poc.user.dto.MemberLoginResponseDto;
import cn.com.poc.user.dto.MemberPasswordRequestDto;
import cn.com.poc.user.entity.MemberInfoEntity;
import cn.com.poc.user.rest.BizMemberInfoRest;
import cn.com.poc.user.service.BizMemberInfoService;
import cn.com.yict.framemax.core.i18n.I18nMessageException;
import cn.com.yict.framemax.security.oauth.OauthAccesstokenManager;
import cn.com.yict.framemax.security.oauth.entity.OauthResultEntity;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;

@Component
public class BizMemberInfoRestImpl implements BizMemberInfoRest {

    @Resource
    private LoginChannelBuilder loginChannelBuilder;

    @Resource
    private OauthAccesstokenManager tokenManager;

    @Resource
    private MemberInfoService memberInfoService;

    @Resource
    private BizMemberInfoService bizMemberInfoService;


    /**
     * 用户登录
     */
    @Override
    public MemberLoginResponseDto doLogin(MemberLoginRequestDto memberLoginRequest) throws Exception {

        Assert.notBlank(memberLoginRequest.getLoginChannel());
        String loginChannel = memberLoginRequest.getLoginChannel();

        MemberInfoEntity memberInfoEntity = (MemberInfoEntity) loginChannelBuilder.getService(loginChannel).doLogin(memberLoginRequest);

        if (memberInfoEntity == null) {
            throw new Exception("登录异常，请稍后再试");
        }

        OauthResultEntity auth = tokenManager.auth(OauthConstants.TypePrefix.MEMBER_DOMAIN + memberInfoEntity.getMemberId(), null, null);

        MemberLoginResponseDto memberLoginResponseDto = new MemberLoginResponseDto();
        memberLoginResponseDto.setMemberId(memberInfoEntity.getMemberId());
        memberLoginResponseDto.setNickName(memberInfoEntity.getNickName());
        memberLoginResponseDto.setAvatarUrl(memberInfoEntity.getAvatarUrl());
        memberLoginResponseDto.setMobilePhone(memberInfoEntity.getMobilePhone());
        memberLoginResponseDto.setToken(auth.getToken());
        return memberLoginResponseDto;
    }

    /**
     * 用户配置密码
     *
     * @param memberPasswordRequestDto
     */
    @Override
    public void configureMemberPassword(MemberPasswordRequestDto memberPasswordRequestDto) throws Exception {
        Assert.notBlank(memberPasswordRequestDto.getAccount());
        Assert.notBlank(memberPasswordRequestDto.getPassword());
        Assert.notBlank(memberPasswordRequestDto.getAuthCode());
        memberInfoService.configureMemberPassword(memberPasswordRequestDto);
    }

    /**
     * 用户更改密码
     */
    @Override
    public void changeMemberPassword(String authCode, String password) throws Exception {
        Assert.notBlank(password);
        Assert.notBlank(authCode);
        UserBaseEntity currentUser = BlContext.getCurrentUser();
        MemberInfoEntity memberInfoEntity = bizMemberInfoService.getById(currentUser.getUserId().intValue());

        MemberPasswordRequestDto memberPasswordRequestDto = new MemberPasswordRequestDto();
        memberPasswordRequestDto.setAccount(memberInfoEntity.getAccount());
        memberPasswordRequestDto.setPassword(password);
        memberPasswordRequestDto.setAuthCode(authCode);
        memberInfoService.configureMemberPassword(memberPasswordRequestDto);
    }


    /**
     * 用户忘记密码
     *
     */
    @Override
    public void forgetMemberPassword(String account, String password, String authCode) throws Exception {
        MemberPasswordRequestDto memberPasswordRequestDto = new MemberPasswordRequestDto();
        memberPasswordRequestDto.setAccount(account);
        memberPasswordRequestDto.setPassword(password);
        memberPasswordRequestDto.setAuthCode(authCode);
        memberInfoService.configureMemberPassword(memberPasswordRequestDto);
    }

    /**
     * 获取用户信息
     *
     * @return
     */
    @Override
    public MemberInfoDto getCurrentMemberInfo() throws Exception {
        MemberInfoEntity memberInfoEntity = bizMemberInfoService.getById(BlContext.getCurrentUser().getUserId().intValue());
        return MemberInfoConvert.converEntity2Dto(memberInfoEntity);
    }


    @Override
    public MemberInfoDto getMemberNickName(Long memberId) throws Exception {
        MemberInfoEntity memberInfoEntity = bizMemberInfoService.getById(memberId.intValue());
        if (memberInfoEntity == null) {
            throw new I18nMessageException("exception/user.does.not.exist");
        }
        MemberInfoDto memberInfoDto = new MemberInfoDto();
        memberInfoDto.setNickName(memberInfoEntity.getNickName());
        memberInfoDto.setAvatarUrl(memberInfoEntity.getAvatarUrl());
        return memberInfoDto;
    }

    @Override
    public MemberInfoDto updateMemberInfo(MemberInfoDto memberInfoDto) {
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        if (userBaseEntity == null) {
            throw new I18nMessageException("exception/user.not.login");
        }
        MemberInfoEntity memberInfoEntity = MemberInfoConvert.convertDto2Entity(memberInfoDto);
        memberInfoEntity.setMemberId(userBaseEntity.getUserId().intValue());
        MemberInfoEntity entity = memberInfoService.updateMemberInfo(memberInfoEntity);
        return MemberInfoConvert.converEntity2Dto(entity);
    }
}