package cn.com.poc.writing.rest.impl;

import cn.com.poc.common.utils.Assert;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.support.security.oauth.entity.UserBaseEntity;
import cn.com.poc.writing.aggregate.ExaminationService;
import cn.com.poc.writing.dto.ExaminationBatchDeleteDto;
import cn.com.poc.writing.dto.ExaminationConfigItemDto;
import cn.com.poc.writing.dto.ExaminationDto;
import cn.com.poc.writing.dto.GenerateExaminationDto;
import cn.com.poc.writing.entity.ExaminationConfigItemEntity;
import cn.com.poc.writing.entity.ExaminationEntity;
import cn.com.poc.writing.entity.GenerateExaminationEntity;
import cn.com.poc.writing.rest.ExaminationRest;
import cn.com.yict.framemax.core.exception.BusinessException;
import cn.hutool.core.bean.BeanUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author 52747
 * @date 2025/8/14
 */
@Component
public class ExaminationRestImpl implements ExaminationRest {

    @Resource
    private ExaminationService examinationService;

    @Override
    public ExaminationConfigItemDto getConfigurationItem() {
        ExaminationConfigItemEntity examinationConfigItemEntity = examinationService.getConfigurationItem();
        ExaminationConfigItemDto dto = new ExaminationConfigItemDto();
        dto.setQuestionType(examinationConfigItemEntity.getQuestionType());
        dto.setLevels(examinationConfigItemEntity.getLevels());
        dto.setLanguages(examinationConfigItemEntity.getLanguages());
        dto.setPositions(examinationConfigItemEntity.getPositions());
        return dto;
    }

    @Override
    public void generateExamination(GenerateExaminationDto generateExaminationDto) {
        Assert.notBlank(generateExaminationDto.getLanguage());
        Assert.notBlank(generateExaminationDto.getTitle());
        Assert.notBlank(generateExaminationDto.getLevel());
        Assert.notEmpty(generateExaminationDto.getQuestionType());
        Assert.notNull(generateExaminationDto.getN());
        if (generateExaminationDto.getN() > 20){
            throw new BusinessException("总题数不可大于20题");
        }
        if (generateExaminationDto.getN() > generateExaminationDto.getTotalScore()){
            throw new BusinessException("总题数不能大于总分数");
        }
        UserBaseEntity currentUser = BlContext.getCurrentUser();
        GenerateExaminationEntity generateExaminationEntity = new GenerateExaminationEntity();
        BeanUtil.copyProperties(generateExaminationDto, generateExaminationEntity);
        examinationService.generateExamination(generateExaminationEntity, currentUser.getUserId());
    }

    @Override
    public ExaminationDto getExaminations(String examinationId) {
        Assert.notBlank(examinationId);
        UserBaseEntity currentUser = BlContext.getCurrentUser();
        ExaminationEntity examinationEntity = examinationService.getExaminations(examinationId, currentUser.getUserId());
        ExaminationDto dto = new ExaminationDto();
        BeanUtil.copyProperties(examinationEntity, dto);
        return dto;
    }

    @Override
    public List<ExaminationDto> getList(String query) {
        List<ExaminationDto> result = new ArrayList<>();
        UserBaseEntity currentUser = BlContext.getCurrentUser();
        List<ExaminationEntity> entities = examinationService.getList(query, currentUser.getUserId());
        if (CollectionUtils.isNotEmpty(entities)) {
            result = entities.stream().map(entity -> {
                ExaminationDto examinationDto = new ExaminationDto();
                examinationDto.setExaminationId(entity.getExaminationId());
                examinationDto.setTitle(entity.getTitle());
                examinationDto.setN(entity.getN());
                examinationDto.setLevel(entity.getLevel());
                examinationDto.setDate(entity.getDate());
                return examinationDto;
            }).collect(Collectors.toList());
        }
        return result;
    }

    @Override
    public void batchDelete(ExaminationBatchDeleteDto dto) {
        Assert.notEmpty(dto.getExaminationIds(), "删除ID不能为空");
        UserBaseEntity currentUser = BlContext.getCurrentUser();
        boolean deleted = examinationService.batchDelete(dto.getExaminationIds(), currentUser.getUserId());
        if (!deleted) {
            throw new BusinessException("删除异常");
        }
    }

    @Override
    public String downloadExamination(String examinationId) {
        Assert.notBlank(examinationId, "examinationId 不能为空");
        UserBaseEntity currentUser = BlContext.getCurrentUser();
        return examinationService.downloadExamination(examinationId, currentUser.getUserId());
    }
}
