package cn.com.poc.common.service;

import cn.com.yict.framemax.core.service.BaseService;
import org.springframework.data.redis.connection.RedisZSetCommands;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Redis工具类
 */
public interface RedisService extends BaseService {


    /****************** common start ****************/
    /**
     * 指定缓存失效时间
     *
     * @param key  键
     * @param time 时间(秒)
     * @return
     */
    public boolean expire(String key, long time);

    /**
     * 根据key 获取过期时间
     *
     * @param key 键 不能为null
     * @return 时间(秒) 返回0代表为永久有效
     */
    public long getExpire(String key);

    /**
     * 判断key是否存在
     *
     * @param key 键
     * @return true 存在 false不存在
     */
    public boolean hasKey(String key);

    /**
     * 删除缓存
     *
     * @param key 可以传一个值 或多个
     */
    public void del(String... key);

    /**
     * 根据key的前缀模糊删除数据
     *
     * @param keyPre redis中key的前缀
     */
    public void delByPre(String keyPre);

    /****************** common end ****************/


    /****************** String start ****************/

    /**
     * 普通缓存获取
     *
     * @param key 键
     * @return 值
     */
    public Object get(String key);

    /**
     * 普通缓存获取 指定序列化对象
     * @param key
     * @param clazz
     * @return
     */
    <T> T get(String key, Class<T> clazz);

    /**
     * 普通缓存放入
     *
     * @param key   键
     * @param value 值
     * @return true成功 false失败
     */
    public boolean set(String key, Object value);

    /**
     * 普通缓存放入并设置时间
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒) time要大于0 如果time小于等于0 将设置无限期
     * @return true成功 false 失败
     */
    public boolean set(String key, Object value, long time);

    /**
     * 递增
     *
     * @param key   键
     * @param delta 要增加几(大于0)
     * @return
     */
    public long incr(String key, long delta);

    /**
     * 递减
     *
     * @param key   键
     * @param delta 要减少几(小于0)
     * @return
     */
    public long decr(String key, long delta);
    /****************** String end ****************/


    /****************** Map start ****************/

    /**
     * HashGet
     *
     * @param key  键 不能为null
     * @param item 项 不能为null
     * @return 值
     */
    public Object hget(String key, String item);

    /**
     * 获取hashKey对应的所有键值
     *
     * @param key 键
     * @return 对应的多个键值
     */
    public Map<Object, Object> hmget(String key);

    /**
     * HashSet
     *
     * @param key 键
     * @param map 对应多个键值
     * @return true 成功 false 失败
     */
    public boolean hmset(String key, Map<String, Object> map);


    /**
     * 批量更新hash
     *
     * @param data 多个hash
     * @param time 过期时间（秒），小于0时为不过期
     * @return 是否成功操作
     */
    public boolean hmSetBatch(Map<String, Map<String, Object>> data, long time);

    /**
     * 根据key前缀批量模糊批量查询hash类型的数据
     *
     * @param keyPre key前缀
     * @return 有则返回数据，无则返回null
     */
    public Map<String, Map<String, Object>> hmDimMultiGet(String keyPre);

    /**
     * HashSet 并设置时间
     *
     * @param key  键
     * @param map  对应多个键值
     * @param time 时间(秒)
     * @return true成功 false失败
     */
    public boolean hmset(String key, Map<String, Object> map, long time);

    /**
     * 向一张hash表中放入数据,如果不存在将创建
     *
     * @param key   键
     * @param item  项
     * @param value 值
     * @return true 成功 false失败
     */
    public boolean hset(String key, String item, Object value);

    /**
     * 向一张hash表中放入数据,如果不存在将创建
     *
     * @param key   键
     * @param item  项
     * @param value 值
     * @param time  时间(秒) 注意:如果已存在的hash表有时间,这里将会替换原有的时间
     * @return true 成功 false失败
     */
    public boolean hset(String key, String item, Object value, long time);

    /**
     * 删除hash表中的值
     *
     * @param key  键 不能为null
     * @param item 项 可以使多个 不能为null
     */
    public void hdel(String key, Object... item);

    /**
     * 判断hash表中是否有该项的值
     *
     * @param key  键 不能为null
     * @param item 项 不能为null
     * @return true 存在 false不存在
     */
    public boolean hHasKey(String key, String item);

    /**
     * hash递增 如果不存在,就会创建一个 并把新增后的值返回
     *
     * @param key  键
     * @param item 项
     * @param by   要增加几(大于0)
     * @return
     */
    public double hincr(String key, String item, long by);

    /**
     * hash递减
     *
     * @param key  键
     * @param item 项
     * @param by   要减少记(小于0)
     * @return
     */
    public double hdecr(String key, String item, long by);


    /****************** Map end ****************/


    /****************** Set start ****************/

    /**
     * 根据key获取Set中的所有值
     *
     * @param key 键
     * @return
     */
    public Set<Object> sGet(String key);

    /**
     * 根据value从一个set中查询,是否存在
     *
     * @param key   键
     * @param value 值
     * @return true 存在 false不存在
     */
    public boolean sHasKey(String key, Object value);


    /**
     * 将数据放入set缓存
     *
     * @param key    键
     * @param values 值 可以是多个
     * @return 成功个数
     */
    public long sSet(String key, Object... values);

    /**
     * 将set数据放入缓存
     *
     * @param key    键
     * @param time   时间(秒)
     * @param values 值 可以是多个
     * @return 成功个数
     */
    public long sSetAndTime(String key, long time, Object... values);


    /**
     * 获取set缓存的长度
     *
     * @param key 键
     * @return
     */
    public long sGetSetSize(String key);


    /**
     * 移除值为value的
     *
     * @param key    键
     * @param values 值 可以是多个
     * @return 移除的个数
     */
    public long setRemove(String key, Object... values);


    /****************** Set end ****************/


    /****************** ZSet start ****************/

    /**
     * @apiNote ZSet存放单个元素
     * @param key 键
     * @param value 值
     * @param score 分数
     * @return
     */
    public Boolean zAdd(String key, Object value, Double score);

    /**
     * @apiNote 往ZSet中指定键批量插入数据
     * @param key ZSet中键
     * @param tuplesMap ZSet中value于score的映射，tuplesMap键为ZSet中的value，值为ZSet中的score
     * @return
     */
    public Long zBatchAdd(String key, Map<Object, Double> tuplesMap);

    /**
     * @apiNote 删除ZSet中指定键的一个或多个value
     * @param key 键
     * @param value ZSet中的一个或多个value
     * @return
     */
    public Long zRemove(String key, Object... value);

    /**
     * @apiNote 获取ZSet指定键的个数
     * @param key 键
     * @return
     */
    public Long zGetSize(String key);

    /**
     * @apiNote 获取ZSet中指定键的指定分数段的value个数
     * @param key 键
     * @param minScore 分数范围左
     * @param maxScore 分数范围右
     * @return
     */
    public Long zGetCount(String key, Double minScore, Double maxScore);

    /**
     * @apiNote 使用索引下标获取ZSet元素, [0,-1]表示全部元素,列表对于score顺数
     * @param key 键
     * @param start 开始索引
     * @param end 结束索引
     * @return
     */
    public Set<RedisZSetCommands.Tuple> zRangeWithScores(String key, Integer start, Integer end);

    /**
     * @apiNote 使用索引下标获取ZSet元素,[0,-1]表示全部元素,列表对于score倒数
     * @param key 键
     * @param start 开始索引
     * @param end 结束索引
     * @return
     */
    public Set<RedisZSetCommands.Tuple> zRevRangeWithScores(String key, Integer start, Integer end);

    /**
     * @apiNote 使用索引下标获取ZSet元素
     * @param key 键
     * @param start 索引开始
     * @param end 索引结束
     * @return
     */
    public Set<Object> zRange(String key, Integer start, Integer end);

    /**
     * @apiNote 使用索引下标获取ZSet元素,并从高到低排序
     * @param key 键
     * @param start 索引开始
     * @param end 索引结束
     * @return
     */
    public Set<Object> zReverseRange(String key, Integer start, Integer end);

    /**
     * @apiNote 获取ZSet中指定键指定value的分数
     * @param key 键
     * @param value ZSet中value
     * @return
     */
    public Double zGetScore(String key, Object value);

    /****************** ZSet end ****************/

    /****************** List start ****************/

    /**
     * 获取list缓存的内容
     *
     * @param key   键
     * @param start 开始
     * @param end   结束 0 到 -1代表所有值
     * @return
     */
    public List<Object> lGet(String key, long start, long end);


    /**
     * 获取list缓存的长度
     *
     * @param key 键
     * @return
     */
    public long lGetListSize(String key);


    /**
     * 通过索引 获取list中的值
     *
     * @param key   键
     * @param index 索引 index>=0时， 0 表头，1 第二个元素，依次类推；index<0时，-1，表尾，-2倒数第二个元素，依次类推
     * @return
     */
    public Object lGetIndex(String key, long index);

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public boolean lSet(String key, Object value);

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒)
     * @return
     */
    public boolean lSet(String key, Object value, long time);

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public boolean lSet(String key, List<Object> value);

    /**
     * 将list放入缓存
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒)
     * @return
     */
    public boolean lSet(String key, List<Object> value, long time);

    /**
     * 根据索引修改list中的某条数据
     *
     * @param key   键
     * @param index 索引
     * @param value 值
     * @return
     */
    public boolean lUpdateIndex(String key, long index, Object value);

    /**
     * 移除N个值为value
     *
     * @param key   键
     * @param count 移除多少个
     * @param value 值
     * @return 移除的个数
     */
    public long lRemove(String key, long count, Object value);

    /**
     *  模糊匹配查询key
     * @param key
     * @return
     */
    public Set<String> keys(String key);
}

