package cn.com.poc.agent_application.aggregate.impl;

import cn.com.poc.agent_application.aggregate.AgentApplicationInfoService;
import cn.com.poc.agent_application.constant.AgentApplicationConstants;
import cn.com.poc.agent_application.constant.AgentApplicationDialoguesRecordConstants;
import cn.com.poc.agent_application.constant.AgentApplicationGCConfigConstants;
import cn.com.poc.agent_application.domain.FunctionResult;
import cn.com.poc.agent_application.entity.*;
import cn.com.poc.agent_application.query.DialogsIdsQueryByAgentIdQueryItem;
import cn.com.poc.agent_application.service.*;
import cn.com.poc.agent_application.utils.AgentApplicationTools;
import cn.com.poc.common.constant.CommonConstant;
import cn.com.poc.common.utils.Assert;
import cn.com.poc.common.utils.BlContext;
import cn.com.poc.common.utils.JsonUtils;
import cn.com.poc.equity.aggregate.MemberEquityService;
import cn.com.poc.equity.domain.MemberEquityInfo;
import cn.com.poc.knowledge.entity.BizKnowledgeDocumentEntity;
import cn.com.poc.knowledge.service.BizKnowledgeDocumentService;
import cn.com.poc.support.security.oauth.entity.UserBaseEntity;
import cn.com.poc.thirdparty.resource.demand.ai.aggregate.AICreateImageService;
import cn.com.poc.thirdparty.resource.demand.ai.aggregate.DemandKnowledgeService;
import cn.com.poc.thirdparty.resource.demand.ai.constants.LLMRoleEnum;
import cn.com.poc.thirdparty.resource.demand.ai.entity.dialogue.Message;
import cn.com.poc.thirdparty.resource.demand.ai.entity.dialogue.MultiContent;
import cn.com.poc.thirdparty.resource.demand.ai.entity.dialogue.Tool;
import cn.com.poc.thirdparty.resource.demand.ai.entity.function.FunctionCallResult;
import cn.com.poc.thirdparty.resource.demand.ai.entity.generations.BaiduAISailsText2ImageRequest;
import cn.com.poc.thirdparty.resource.demand.ai.entity.generations.BaiduAISailsText2ImageResult;
import cn.com.poc.thirdparty.resource.demand.ai.entity.largemodel.LargeModelDemandResult;
import cn.com.poc.thirdparty.resource.demand.ai.entity.largemodel.LargeModelResponse;
import cn.com.poc.thirdparty.resource.demand.ai.function.LargeModelFunctionEnum;
import cn.com.poc.thirdparty.resource.demand.ai.function.long_memory.LongMemory;
import cn.com.poc.thirdparty.resource.demand.ai.function.long_memory.AgentLongMemoryEntity;
import cn.com.poc.thirdparty.resource.demand.ai.function.memory_variable_writer.MemoryVariableWriter;
import cn.com.poc.thirdparty.service.LLMService;
import cn.com.yict.framemax.core.context.Context;
import cn.com.yict.framemax.core.i18n.I18nMessageException;
import cn.com.yict.framemax.data.model.PagingInfo;
import cn.hutool.core.bean.BeanUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.*;

@Component
public class AgentApplicationInfoServiceImpl implements AgentApplicationInfoService {

    final private String TEXT_EVENT_STREAM_CHARSET_UTF_8 = "text/event-stream;charset=UTF-8";

    final private String EVENT_STREAM_PREFIX = "data: ";

    final private Logger logger = LoggerFactory.getLogger(AgentApplicationInfoService.class);

    @Resource
    private BizKnowledgeDocumentService bizKnowledgeDocumentService;

    @Resource
    private BizAgentApplicationLargeModelListService bizAgentApplicationLargeModelListService;

    @Resource
    private BizAgentApplicationGcConfigService bizAgentApplicationGcConfigService;

    @Resource
    private BizAgentApplicationInfoService bizAgentApplicationInfoService;

    @Resource
    private BizAgentApplicationPublishService bizAgentApplicationPublishService;

    @Resource
    private DemandKnowledgeService demandKnowledgeService;

    @Resource
    private AICreateImageService aiCreateImageService;

    @Resource
    private LLMService llmService;

    @Resource
    private BizAgentApplicationMallService bizAgentApplicationMallService;

    @Resource
    private BizMemberAgentApplicationCollectService bizMemberAgentApplicationCollectService;

    @Resource
    private BizAgentApplicationDialoguesRecordService bizAgentApplicationDialoguesRecordService;


    @Override
    public BizAgentApplicationInfoEntity saveOrUpdate(BizAgentApplicationInfoEntity entity) throws Exception {

        // 如果存在agentId，则判断变量结构是否有变化，如果有变化，则删除redis中的数据
        if (StringUtils.isNotBlank(entity.getAgentId())) {
            BizAgentApplicationInfoEntity infoEntity = bizAgentApplicationInfoService.getByAgentId(entity.getAgentId());
            updateStructVariable(entity.getAgentId(), infoEntity.getVariableStructure(), entity.getVariableStructure());
        }
        // 保存或更新
        return StringUtils.isEmpty(entity.getAgentId()) ?
                bizAgentApplicationInfoService.save(entity) : bizAgentApplicationInfoService.update(entity);
    }

    @Override
    public boolean updateAndPublish(BizAgentApplicationInfoEntity bizAgentApplicationInfoEntity) throws Exception {
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        Long userId = userBaseEntity.getUserId();
        bizAgentApplicationInfoEntity.setMemberId(userId.intValue());
        bizAgentApplicationInfoEntity.setAgentPublishStatus(AgentApplicationConstants.AGENT_PUBLISH_STATUS.PUBLISH);
        bizAgentApplicationInfoEntity.setPublishTime(new Date());
        if (null == bizAgentApplicationInfoService.update(bizAgentApplicationInfoEntity)) {
            logger.error("update and publish agent error , bizAgentApplicationInfoEntity:{}", bizAgentApplicationInfoEntity);
            throw new I18nMessageException("exception/publication.failed");
        }

        //发布应用
        BizAgentApplicationPublishEntity publishEntity = new BizAgentApplicationPublishEntity();
        BeanUtil.copyProperties(bizAgentApplicationInfoEntity, publishEntity);
        BizAgentApplicationPublishEntity oldPublishEntity = bizAgentApplicationPublishService.getByAgentId(bizAgentApplicationInfoEntity.getAgentId());
        if (oldPublishEntity != null) {
            publishEntity.setId(oldPublishEntity.getId());
            bizAgentApplicationPublishService.update(publishEntity);
        } else {
            bizAgentApplicationPublishService.save(publishEntity);
        }

        //【变量】结构改变，删除会话缓存
        List<DialogsIdsQueryByAgentIdQueryItem> items = bizAgentApplicationDialoguesRecordService.queryDialogsIds(bizAgentApplicationInfoEntity.getAgentId());
        if (CollectionUtils.isNotEmpty(items)) {
            logger.info("variables structure cache remove，items:{}", items);
            for (DialogsIdsQueryByAgentIdQueryItem item : items) {
                MemoryVariableWriter.cleanByPre(AgentApplicationTools.identifier(item.getDialogsId(), bizAgentApplicationInfoEntity.getAgentId()));
            }
        }

        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deletedAgentApplication(String agentId) throws Exception {
        bizAgentApplicationInfoService.deletedByAgentId(agentId);
        // 删除应用
        BizAgentApplicationPublishEntity publishEntity = bizAgentApplicationPublishService.getByAgentId(agentId);
        if (publishEntity != null) {
            // 删除应用市场表中的应用
            bizAgentApplicationMallService.deletedByAgentPublishId(publishEntity.getId());
            // 删除发布表中的应用
            bizAgentApplicationPublishService.deleteByAgentId(agentId);
        }
        return true;
    }

    @Override
    public String callAgentApplication(String agentId, String dialogueId, String largeModel,
                                       String agentSystem, Integer[] kdIds, Integer communicationTurn, Float topP, Float temperature,
                                       List<Message> messages, List<Tool> tools, List<String> fileUrls, HttpServletResponse httpServletResponse) throws Exception {
        logger.info("Call Agent Application, agentId:{}, dialogueId:{},largeModel:{},agentSystem:{},kdIds:{},communicationTurn:{},topP:{},temperature:{},messages:{}, tools:{}"
                , agentId, dialogueId, largeModel, agentSystem, kdIds, communicationTurn, topP, temperature, messages, tools);

        String model = modelConvert(largeModel);

        Tool[] toolArray = tools.toArray(new Tool[0]);

        FunctionResult functionResult = functionCall(dialogueId, messages, toolArray, agentId, fileUrls);

        String promptTemplate = buildDialogsPrompt(functionResult, messages, agentSystem, kdIds, toolArray, dialogueId, agentId);

        Message[] messageArray = buildMessages(messages, communicationTurn, promptTemplate);

        BufferedReader bufferedReader = invokeLLM(model, messageArray, topP);

        return textOutput(httpServletResponse, bufferedReader);
    }


    @Override
    public void createAgentSystem(String input, HttpServletResponse httpServletResponse) throws Exception {
        BizAgentApplicationGcConfigEntity configEntity = bizAgentApplicationGcConfigService.getByConfigCode(AgentApplicationGCConfigConstants.AGENT_SYSTEM);
        if (null == configEntity || StringUtils.isBlank(configEntity.getConfigSystem())) {
            throw new I18nMessageException("exception/create.[role.instruction].configuration.does.not.exist");
        }
        Message systemMessage = new Message();
        systemMessage.setContent(configEntity.getConfigSystem());
        systemMessage.setRole(LLMRoleEnum.SYSTEM.getRole());

        Message message = new Message();
        message.setContent("{" + input + "}");
        message.setRole(LLMRoleEnum.USER.getRole());
        List<Message> messages = new ArrayList<Message>() {{
            add(systemMessage);
            add(message);
        }};

        LargeModelResponse largeModelResponse = new LargeModelResponse();
        largeModelResponse.setModel(configEntity.getLargeModel());
        largeModelResponse.setMessages(messages.toArray(new Message[1]));
        largeModelResponse.setTopP(configEntity.getTopP());
        largeModelResponse.setUser("POC-CREATE-AGENT-SYSTEM");

        BufferedReader bufferedReader = llmService.chatChunk(largeModelResponse);
        textOutput(httpServletResponse, bufferedReader);
    }

    @Override
    public String createPreamble(String agentTitle, String agentDesc, String agentSystem) {
        BizAgentApplicationGcConfigEntity configEntity = bizAgentApplicationGcConfigService.getByConfigCode(AgentApplicationGCConfigConstants.AGENT_PREAMBLE);
        if (null == configEntity || StringUtils.isBlank(configEntity.getConfigSystem())) {
            throw new I18nMessageException("exception/create.[opening.statement].configuration.does.not.exist");
        }

        List<MultiContent> multiContents = new ArrayList<>();
        multiContents.add(new MultiContent() {{
            String configSystem = configEntity.getConfigSystem();
            configSystem = buildPreambleConfigSystem(configSystem, agentTitle, agentDesc, agentSystem);

            setText(configSystem);
            setType("text");
        }});
        Message message = new Message();
        message.setContent(multiContents);
        message.setRole(LLMRoleEnum.USER.getRole());
        List<Message> messages = new ArrayList<Message>() {{
            add(message);
        }};

        LargeModelResponse largeModelResponse = new LargeModelResponse();
        largeModelResponse.setModel(configEntity.getLargeModel());
        largeModelResponse.setMessages(messages.toArray(new Message[1]));
        largeModelResponse.setTopP(configEntity.getTopP());
        largeModelResponse.setUser("POC-CREATE-PREAMBLE");

        LargeModelDemandResult largeModelDemandResult = llmService.chat(largeModelResponse);
        if (largeModelDemandResult == null || !"0".equals(largeModelDemandResult.getCode())) {
            logger.error("create preamble error, largeModelResponse:{}", largeModelResponse);
            throw new I18nMessageException("exception/failed.to.create.[opening.statement]");
        }

        return largeModelDemandResult.getMessage();
    }

    @Override
    public List<String> createFeaturedQuestions(String agentTitle, String agentDesc) {
        BizAgentApplicationGcConfigEntity configEntity = bizAgentApplicationGcConfigService.getByConfigCode(AgentApplicationGCConfigConstants.AGENT_FEATURED_QUESTIONS);
        if (null == configEntity || StringUtils.isBlank(configEntity.getConfigSystem())) {
            throw new I18nMessageException("exception/create.[opening.statement].configuration.does.not.exist");
        }

        List<MultiContent> multiContents = new ArrayList<>();
        multiContents.add(new MultiContent() {{
            String configSystem = configEntity.getConfigSystem();
            configSystem = configSystem.replace("${agent_title}", StringUtils.isNotBlank(agentTitle) ? agentTitle : StringUtils.EMPTY);
            configSystem = configSystem.replace("${agent_desc}", StringUtils.isNotBlank(agentDesc) ? agentDesc : StringUtils.EMPTY);
            setText(configSystem);
            setType("text");
        }});
        Message message = new Message();
        message.setContent(multiContents);
        message.setRole(LLMRoleEnum.USER.getRole());
        List<Message> messages = new ArrayList<Message>() {{
            add(message);
        }};

        LargeModelResponse largeModelResponse = new LargeModelResponse();
        largeModelResponse.setModel(configEntity.getLargeModel());
        largeModelResponse.setMessages(messages.toArray(new Message[1]));
        largeModelResponse.setTopP(configEntity.getTopP());
        largeModelResponse.setUser("POC-FEATURED-QUESTIONS");

        LargeModelDemandResult largeModelDemandResult = llmService.chat(largeModelResponse);
        String res = largeModelDemandResult.getMessage();
        int start = res.lastIndexOf("[");
        int end = res.lastIndexOf("]");
        return JsonUtils.deSerialize(res.substring(start, end + 1), new TypeReference<List<String>>() {
        }.getType());
    }

    @Override
    public String createAgentIcon(String agentTitle, String agentDesc) throws Exception {

        BizAgentApplicationGcConfigEntity configEntity = bizAgentApplicationGcConfigService.getByConfigCode(AgentApplicationGCConfigConstants.AGENT_ICON);
        if (null == configEntity || StringUtils.isBlank(configEntity.getConfigSystem())) {
            throw new I18nMessageException("exception/the.configuration.for.creating.[avatar.prompt].does.not.exist.please.contact.the.administrator");
        }

        List<MultiContent> multiContents = new ArrayList<>();
        multiContents.add(new MultiContent() {{
            String configSystem = configEntity.getConfigSystem();
            configSystem = configSystem.replace("${agent_title}", StringUtils.isNotBlank(agentTitle) ? agentTitle : StringUtils.EMPTY);
            configSystem = configSystem.replace("${agent_desc}", StringUtils.isNotBlank(agentDesc) ? agentDesc : StringUtils.EMPTY);
            setText(configSystem);
            setType("text");
        }});
        Message message = new Message();
        message.setContent(multiContents);
        message.setRole(LLMRoleEnum.USER.getRole());
        List<Message> messages = new ArrayList<Message>() {{
            add(message);
        }};

        LargeModelResponse largeModelResponse = new LargeModelResponse();
        largeModelResponse.setModel(configEntity.getLargeModel());
        largeModelResponse.setMessages(messages.toArray(new Message[1]));
        largeModelResponse.setTopP(configEntity.getTopP());
        largeModelResponse.setUser("POC-CREATE-AGENT-PROMPT");
        LargeModelDemandResult largeModelDemandResult = llmService.chat(largeModelResponse);

        if (largeModelDemandResult == null || !"0".equals(largeModelDemandResult.getCode()) || StringUtils.isEmpty(largeModelDemandResult.getMessage())) {
            logger.error("create agent icon prompt error, largeModelResponse:{}", largeModelResponse);
            throw new I18nMessageException("exception/failed.to.create.[avatar].please.try.again.later");
        }

        logger.info("create agent application icon prompt, result:{}", largeModelDemandResult);

        String createIconPrompt = largeModelDemandResult.getMessage();
        BaiduAISailsText2ImageRequest baiduAISailsText2ImageRequest = new BaiduAISailsText2ImageRequest();
        baiduAISailsText2ImageRequest.setPrompt(createIconPrompt);
        baiduAISailsText2ImageRequest.setNegativePrompt(null);
        baiduAISailsText2ImageRequest.setSize("768x768");
        baiduAISailsText2ImageRequest.setN(1);
        baiduAISailsText2ImageRequest.setSteps(15);
        baiduAISailsText2ImageRequest.setSamplerIndex("Euler a");
        baiduAISailsText2ImageRequest.setSeed(null);
        baiduAISailsText2ImageRequest.setCfgCcale(null);
        baiduAISailsText2ImageRequest.setStyle("Base");
        baiduAISailsText2ImageRequest.setUserId("AGENT-APPLICATION-ICON-CREATE");
        baiduAISailsText2ImageRequest.setAcctId(null);

        BaiduAISailsText2ImageResult baiduAISailsText2ImageResult = aiCreateImageService.executeSailsText2Image(baiduAISailsText2ImageRequest);
        if (baiduAISailsText2ImageResult == null
                || CollectionUtils.isEmpty(baiduAISailsText2ImageResult.getData())
                || StringUtils.isEmpty(baiduAISailsText2ImageResult.getData().get(0).getUrl())) {
            logger.error("create agent icon error, baiduAISailsText2ImageRequest:{} ", baiduAISailsText2ImageRequest);
            throw new I18nMessageException("exception/failed.to.create.[avatar].please.try.again.later");
        }
        return baiduAISailsText2ImageResult.getData().get(0).getUrl();
    }

    @Override
    public CreateAgentTitleAndDescEntity createAgentTitleAndDesc(String input) {
        BizAgentApplicationGcConfigEntity configEntity = bizAgentApplicationGcConfigService.getByConfigCode(AgentApplicationGCConfigConstants.AGENT_INFO);
        if (null == configEntity || StringUtils.isBlank(configEntity.getConfigSystem())) {
            throw new I18nMessageException("exception/create.[application.information].configuration.does.not.exist.please.contact.the.administrator");
        }
        List<MultiContent> multiContents = new ArrayList<>();
        multiContents.add(new MultiContent() {{
            String configSystem = configEntity.getConfigSystem();
            configSystem = configSystem.replace("${input}", StringUtils.isNotBlank(input) ? input : StringUtils.EMPTY);
            setText(configSystem);
            setType("text");
        }});
        Message message = new Message();
        message.setContent(multiContents);
        message.setRole(LLMRoleEnum.USER.getRole());
        List<Message> messages = new ArrayList<Message>() {{
            add(message);
        }};

        LargeModelResponse largeModelResponse = new LargeModelResponse();
        largeModelResponse.setModel(configEntity.getLargeModel());
        largeModelResponse.setMessages(messages.toArray(new Message[1]));
        largeModelResponse.setTopP(configEntity.getTopP());
        largeModelResponse.setUser("POC-CREATE-AGENT-INFO");
        LargeModelDemandResult largeModelDemandResult = llmService.chat(largeModelResponse);

        if (largeModelDemandResult == null || !"0".equals(largeModelDemandResult.getCode()) || StringUtils.isEmpty(largeModelDemandResult.getMessage())) {
            logger.error("create agent icon title And desc error, largeModelResponse:{}", largeModelResponse);
            throw new I18nMessageException("exception/failed.to.create.[application.information].please.try.again.later");
        }
        logger.info("create agent application icon title And desc, response:{},result:{}", largeModelResponse, largeModelDemandResult);
        String res = largeModelDemandResult.getMessage();
        int start = res.lastIndexOf("{");
        int end = res.lastIndexOf("}");

        return JsonUtils.deSerialize(res.substring(start, end + 1), CreateAgentTitleAndDescEntity.class);
    }

    @Override
    public boolean unPublish(String agentId) throws Exception {
        bizAgentApplicationPublishService.deleteByAgentId(agentId);
        bizAgentApplicationInfoService.unPublish(agentId);
        return true;
    }

    @Override
    public void collectOrCancelAgentInMall(Integer id, String agentId) throws Exception {
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        if (userBaseEntity == null) {
            throw new I18nMessageException("exception/user.not.login");
        }

        BizAgentApplicationMallEntity mallEntity = bizAgentApplicationMallService.get(id);
        if (mallEntity == null) {
            throw new I18nMessageException("exception/application.does.not.exist");
        }

        // 先判断当前应用被当前用户收藏了吗
        BizMemberAgentApplicationCollectEntity collectEntity = new BizMemberAgentApplicationCollectEntity();
        collectEntity.setAgentId(agentId);
        collectEntity.setMemberId(userBaseEntity.getUserId().intValue());
        List<BizMemberAgentApplicationCollectEntity> collectEntities = bizMemberAgentApplicationCollectService.findByExample(collectEntity, new PagingInfo());
        if (collectEntities.isEmpty() || CommonConstant.IsDeleted.N.equals(collectEntities.get(0).getIsCollect())) {   //如果没被收藏
            // 收藏该应用
            // 说明从来没收藏过
            if (collectEntities.isEmpty()) {
                // 在用户收藏表中添加数据
                BizMemberAgentApplicationCollectEntity entity = new BizMemberAgentApplicationCollectEntity();
                entity.setIsCollect(CommonConstant.IsDeleted.Y);
                entity.setMemberId(userBaseEntity.getUserId().intValue());
                entity.setAgentId(agentId);
                bizMemberAgentApplicationCollectService.save(entity);
            } else {  // 说明之前收藏过，则只需要做个修改
                collectEntity.setIsCollect(CommonConstant.IsDeleted.Y);
                bizMemberAgentApplicationCollectService.update(collectEntity);
            }
            // 在应用广场的应用中 添加收藏数
            mallEntity.setCollectNumber(mallEntity.getCollectNumber() + 1);
        } else {
            // 取消收藏该应用
            collectEntities.get(0).setIsCollect(CommonConstant.IsDeleted.N);
            bizMemberAgentApplicationCollectService.update(collectEntity);
            // 在应用广场的应用中 取消收藏数
            mallEntity.setCollectNumber(mallEntity.getCollectNumber() - 1);
        }
        bizAgentApplicationMallService.update(mallEntity);
    }

    @Override
    public void collectOrCancelAgentInPerson(String agentId) throws Exception {
        UserBaseEntity userBaseEntity = BlContext.getCurrentUserNotException();
        if (userBaseEntity == null) {
            throw new I18nMessageException("exception/user.not.login");
        }

        // 先判断当前应用被当前用户收藏了吗
        BizMemberAgentApplicationCollectEntity collectEntity = new BizMemberAgentApplicationCollectEntity();
        collectEntity.setAgentId(agentId);
        collectEntity.setMemberId(userBaseEntity.getUserId().intValue());
        List<BizMemberAgentApplicationCollectEntity> collectEntities = bizMemberAgentApplicationCollectService.findByExample(collectEntity, new PagingInfo());
        Integer isCollect = 0;
        if (collectEntities.isEmpty() || CommonConstant.IsDeleted.N.equals(collectEntities.get(0).getIsCollect())) {   //如果没被收藏
            // 收藏该应用
            // 说明从来没收藏过
            if (collectEntities.isEmpty()) {
                // 在用户收藏表中添加数据
                BizMemberAgentApplicationCollectEntity entity = new BizMemberAgentApplicationCollectEntity();
                entity.setIsCollect(CommonConstant.IsDeleted.Y);
                entity.setMemberId(userBaseEntity.getUserId().intValue());
                entity.setAgentId(agentId);
                bizMemberAgentApplicationCollectService.save(entity);
            } else {  // 说明之前收藏过，则只需要做个修改
                collectEntities.get(0).setIsCollect(CommonConstant.IsDeleted.Y);
                bizMemberAgentApplicationCollectService.update(collectEntities.get(0));
            }
            isCollect = 1;
        } else {
            // 取消收藏该应用
            collectEntities.get(0).setIsCollect(CommonConstant.IsDeleted.N);
            bizMemberAgentApplicationCollectService.update(collectEntities.get(0));
            isCollect = -1;
        }

        BizAgentApplicationPublishEntity publishEntity = bizAgentApplicationPublishService.getByAgentId(agentId);
        if (publishEntity == null) {
            logger.warn("应用尚未发布,agentId:{}", agentId);
            return;
        }
        Integer id = publishEntity.getId();
        BizAgentApplicationMallEntity mallEntity = bizAgentApplicationMallService.getByAgentPublishId(id);
        if (mallEntity == null || CommonConstant.YOrN.N.equals(mallEntity.getIsSale())) {
            logger.warn("应用尚未上架,agentId:{}", agentId);
            return;
        }
        mallEntity.setCollectNumber(mallEntity.getCollectNumber() + isCollect);
        bizAgentApplicationMallService.update(mallEntity);
    }


    public boolean hasPublishAgentPermission(String agentId, Long memberId) {
        BizAgentApplicationPublishEntity publishEntity = bizAgentApplicationPublishService.getByAgentId(agentId);
        if (publishEntity != null && publishEntity.getMemberId().equals(memberId.intValue())) {
            return true;
        }
        return false;
    }

    /**
     * 构建应用信息提示词
     *
     * @param configSystem
     * @param agentTitle
     * @param agentDesc
     * @param agentSystem
     * @return
     */
    private String buildPreambleConfigSystem(String configSystem, String agentTitle, String agentDesc, String agentSystem) {
        //若 name  desc    prompt  都为空，传入默认参数  name  入参 智能问答机器人   desc  随时为你解答问题
        if (StringUtils.isBlank(agentSystem) && StringUtils.isBlank(agentTitle) && StringUtils.isBlank(agentDesc)) {
            return configSystem.replace("${agent_title}", "智能问答机器人").replace("${agent_desc}", " 随时为你解答问题");
        }
        //若 name  desc   其中一项不为空则入参  不传入prompt
        if (StringUtils.isNotBlank(agentTitle) || (StringUtils.isNotBlank(agentDesc))) {
            return configSystem.replace("${agent_title}", StringUtils.isNotBlank(agentTitle) ? agentTitle : StringUtils.EMPTY)
                    .replace("${agent_desc}", StringUtils.isNotBlank(agentDesc) ? agentDesc : StringUtils.EMPTY);
        }

        // 若 name  desc 都空   prompt 有内容，则把prompt 传入desc
        if (StringUtils.isNotBlank(agentSystem)) {
            return configSystem.replace("${agent_desc}", agentSystem).replace("${agent_title}", "智能问答机器人");
        }
        return configSystem;
    }


    /**
     * 构建对话提示词
     *
     * @param functionResult 函数结果
     * @param messages       对话消息
     * @param agentSystem    应用角色指令
     * @param kdIds          知识库id
     * @param tools          组件
     * @param dialogueId     对话标识符
     * @return
     */
    private String buildDialogsPrompt(FunctionResult functionResult, List<Message> messages, String agentSystem, Integer[] kdIds, Tool[] tools, String dialogueId, String agentId) {
        String promptTemplate = bizAgentApplicationGcConfigService.getByConfigCode(AgentApplicationGCConfigConstants.AGENT_BASE_SYSTEM).getConfigSystem();
        Locale currentLocale = Context.get().getMessageSource().getCurrentLocale();
        // 系统语言
        promptTemplate = promptTemplate.replace("${language}", currentLocale.toLanguageTag());
        // 应用角色指令
        promptTemplate = promptTemplate.replace("${agentSystem}", StringUtils.isNotBlank(agentSystem) ? agentSystem : StringUtils.EMPTY);
        // 调用知识库
        promptTemplate = buildKnowledgePrompt(messages, kdIds, promptTemplate);
        // 记忆
        promptTemplate = buildMemoryPrompt(promptTemplate, tools, dialogueId, agentId);
        // 函数调用
        promptTemplate = buildFunctionPrompt(functionResult, promptTemplate);
        return promptTemplate;
    }

    /**
     * 构建知识库提示词
     *
     * @param messages
     * @param kdIds
     * @param promptTemplate
     * @return
     */
    private String buildKnowledgePrompt(List<Message> messages, Integer[] kdIds, String promptTemplate) {
        if (ArrayUtils.isNotEmpty(kdIds)) {
            List<String> knowledgeIds = new ArrayList<>();
            for (Integer kdId : kdIds) {
                BizKnowledgeDocumentEntity knowledgeDocumentEntity = bizKnowledgeDocumentService.get(kdId);
                if (null == knowledgeDocumentEntity) {
                    continue;
                }
                knowledgeIds.add(knowledgeDocumentEntity.getKnowledgeId());
            }
            Object content = messages.get(messages.size() - 1).getContent();
            String query = "";
            if (content instanceof List) {
                query = ((List<HashMap>) content).get(0).get("text").toString();
            } else {
                query = content.toString();
            }
            List<String> knowledgeResults = demandKnowledgeService.searchKnowledge(query, knowledgeIds, 3);
            StringBuilder knowledgeResultsBuilder = new StringBuilder();
            if (CollectionUtils.isNotEmpty(knowledgeResults)) {
                for (int i = 1; i <= knowledgeResults.size(); i++) {
                    knowledgeResultsBuilder.append("### Chunk ").append(i).append(":").append(StringUtils.LF).append(knowledgeResults.get(i - 1)).append(StringUtils.LF);
                }
            }
            promptTemplate = promptTemplate.replace("${knowledgeResults}", knowledgeResultsBuilder.toString());
        }
        return promptTemplate;
    }

    /**
     * 构造记忆体提示词
     *
     * @param promptTemplate
     * @param tools
     * @param dialogueId
     * @param agentId
     * @return
     */
    private String buildMemoryPrompt(String promptTemplate, Tool[] tools, String dialogueId, String agentId) {
        String longMemoryContent = StringUtils.EMPTY;
        String valueMemoryContent = StringUtils.EMPTY;
        if (ArrayUtils.isNotEmpty(tools)) {
            for (Tool tool : tools) {
                String name = tool.getFunction().getName();
                // 长期记忆
                if (LargeModelFunctionEnum.set_long_memory.name().equals(name)) {
                    List<AgentLongMemoryEntity> longMemoryEntities = LongMemory.get(AgentApplicationTools.identifier(dialogueId, agentId));
                    if (CollectionUtils.isNotEmpty(longMemoryEntities)) {
                        StringBuilder stringBuilder = new StringBuilder("");
                        for (AgentLongMemoryEntity agentLongMemoryEntity : longMemoryEntities) {
                            stringBuilder
                                    .append("Time").append(":").append(agentLongMemoryEntity.getTimestamp())
                                    .append(StringUtils.SPACE)
                                    .append("Content").append(":").append(agentLongMemoryEntity.getContent())
                                    .append(StringUtils.LF);
                        }
                        longMemoryContent = stringBuilder.toString();
                    }
                }

                // 变量记忆
                if (LargeModelFunctionEnum.memory_variable_writer.name().equals(name)) {
                    Map<Object, Object> map = MemoryVariableWriter.get(AgentApplicationTools.identifier(dialogueId, agentId));
                    StringBuilder stringBuilder = new StringBuilder("");
                    if (MapUtils.isNotEmpty(map)) {
                        Set<Object> keySet = map.keySet();
                        for (Object key : keySet) {
                            stringBuilder.append(key.toString()).append(":").append(map.get(key)).append(StringUtils.LF);
                        }
                    }
                    valueMemoryContent = stringBuilder.toString();
                }
            }
        }
        return promptTemplate.replace("${longMemoryResult}", longMemoryContent).replace("${valueMemoryResult}", valueMemoryContent);
    }

    /**
     * 构建函数调用提示词
     *
     * @param functionResult
     * @param promptTemplate
     * @return
     */
    private String buildFunctionPrompt(FunctionResult functionResult, String promptTemplate) {
        if (functionResult != null) {
            promptTemplate = promptTemplate.replace("${functionName}", StringUtils.isNotBlank(functionResult.getFunctionName()) ? functionResult.getFunctionName() : StringUtils.EMPTY)
                    .replace("${functionArg}", StringUtils.isNotBlank(functionResult.getFunctionArg()) ? functionResult.getFunctionArg() : StringUtils.EMPTY)
                    .replace("${functionDesc}", StringUtils.isNotBlank(functionResult.getFunctionDesc()) ? functionResult.getFunctionDesc() : StringUtils.EMPTY)
                    .replace("${functionResult}", StringUtils.isNotBlank(functionResult.getFunctionResult()) ? functionResult.getFunctionResult() : StringUtils.EMPTY);
        }
        return promptTemplate;
    }

    /**
     * * 调用llm
     *
     * @param largeModel
     * @param messageArray
     * @param topP
     * @return
     * @throws Exception
     */
    private BufferedReader invokeLLM(String largeModel, Message[] messageArray, Float topP) throws Exception {
        LargeModelResponse largeModelResponse = new LargeModelResponse();
        largeModelResponse.setModel(largeModel);
        largeModelResponse.setMessages(messageArray);
        largeModelResponse.setTopP(topP);
        largeModelResponse.setStream(true);
        largeModelResponse.setUser("POE");
        return llmService.chatChunk(largeModelResponse);
    }


    /**
     * 文本输出结果
     *
     * @param httpServletResponse
     * @param bufferedReader
     * @throws IOException
     */
    private String textOutput(HttpServletResponse httpServletResponse, BufferedReader bufferedReader) throws
            IOException {
        String res = "";
        httpServletResponse.setContentType(TEXT_EVENT_STREAM_CHARSET_UTF_8);
        PrintWriter writer = httpServletResponse.getWriter();
        StringBuilder output = new StringBuilder();
        while ((res = bufferedReader.readLine()) != null) {
            if (StringUtils.isBlank(res)) {
                continue;
            }
            LargeModelDemandResult result = JsonUtils.deSerialize(res.replaceFirst(EVENT_STREAM_PREFIX, StringUtils.EMPTY), LargeModelDemandResult.class);
            if (!"0".equals(result.getCode())) {
                logger.error("LLM Error,code:{}", result.getCode());
                I18nMessageException ex = new I18nMessageException("exception/call.failure");
                writer.write(EVENT_STREAM_PREFIX + JsonUtils.serialize(ex) + "\n\n");
                writer.flush();
                throw ex;
            }
            if (StringUtils.isNotBlank(result.getMessage())) {
                output.append(result.getMessage());
            }
            writer.write(res + "\n\n");
            writer.flush();
        }
        writer.write(EVENT_STREAM_PREFIX + "[DONE]\n\n");
        // 关闭资源
        bufferedReader.close();
        writer.close();
        return output.toString();
    }

    /**
     * 构建消息【大模型参数】
     *
     * @param messages
     * @param communicationTurn
     * @param promptTemplate
     * @return
     */
    private Message[] buildMessages(List<Message> messages, Integer communicationTurn, String promptTemplate) {
        int messLength = messages.size() - 1;
        int skip = communicationTurn * 2;
        if (skip < messLength) {
            messages = messages.subList(messLength - skip, messages.size());
        }

        if (StringUtils.isNotBlank(promptTemplate)) {
            Message systemMessage = new Message();
            systemMessage.setRole(AgentApplicationDialoguesRecordConstants.ROLE.SYSTEM);
            systemMessage.setContent(promptTemplate);
            messages.add(0, systemMessage);
        }
        // 转换消息体
        Message[] messageArray = new Message[messages.size()];
        int index = 0;
        for (Message message : messages) {
            if (message.getContent() instanceof List) {
                List<HashMap<String, Object>> content = (List<HashMap<String, Object>>) message.getContent();
                HashMap<String, Object> map = content.get(0);
                if (map.containsKey("type") && map.get("type").equals("text") && map.containsKey("text")) {
                    message.setContent(map.get("text").toString());
                }
            }
            messageArray[index++] = message;
        }
        return messageArray;
    }


    /**
     * 获取模型
     *
     * @param largeModel
     * @return
     */
    private String modelConvert(String largeModel) {
        BizAgentApplicationLargeModelListEntity largeModelEntity = bizAgentApplicationLargeModelListService.findByModelNickName(largeModel);
        if (largeModelEntity == null) {
            throw new I18nMessageException("模型不存在");
        }
        return largeModelEntity.getModelName();
    }

    /**
     * 判断是否需要FunctionCall
     *
     * @param dialogueId
     * @param messages
     * @param tools
     * @param agentId
     */
    private FunctionResult functionCall(String dialogueId, List<Message> messages, Tool[] tools, String agentId, List<String> fileUrls) {
        FunctionResult result = new FunctionResult();
        if (ArrayUtils.isEmpty(tools)) {
            return result;
        }
        Object content = messages.get(messages.size() - 1).getContent();
        String query;
        if (content instanceof List) {
            query = ((List<HashMap>) content).get(0).get("text").toString();
        } else {
            query = content.toString();
        }

        if (CollectionUtils.isNotEmpty(fileUrls)) {
            query = JsonUtils.serialize(fileUrls) + query;
        }

        FunctionCallResult functionCallResult = llmService.functionCall(query, tools);
        if (functionCallResult != null && functionCallResult.isNeed()) {
            // 执行函数返回结果
            LargeModelFunctionEnum functionEnum = LargeModelFunctionEnum.valueOf(functionCallResult.getFunctionCall().getName());
            String functionResult = functionEnum.getFunction().doFunction(functionCallResult.getFunctionCall().getArguments(), AgentApplicationTools.identifier(dialogueId, agentId));

            //构造返回结果
            result.setFunctionName(functionCallResult.getFunctionCall().getName());
            result.setFunctionArg(functionCallResult.getFunctionCall().getArguments());
            result.setFunctionDesc(functionEnum.getFunction().getDesc());
            result.setFunctionResult(functionResult);
        }
        return result;

    }


    /**
     * 更新【记忆变量】结构
     *
     * @param agentId     应用ID
     * @param originals   原字段结构
     * @param transformed 变换后的字段结构
     * @author alex.yao
     * @deprecated 更新规则:
     * 1. 原字段已存在，并存在值则不更新，否则更新并赋予该字段新结构默认值
     * 2. 原字段不存在，则新增并赋予新结构的默认值
     * 3. 原字段存在，但新结构中不存在，则删除原字段
     * 注意: 若transformed为空会清除所有字段，删除对话中所有【记忆变量】缓存
     */
    private void updateStructVariable(String agentId, List<Variable> originals, List<Variable> transformed) {
        Assert.notBlank(agentId);
        String identifier = AgentApplicationTools.identifier(agentId, agentId);
        if (CollectionUtils.isEmpty(transformed)) {
            // 清除所有字段
            MemoryVariableWriter.clean(identifier);
        }
        // 原【变量记忆】为空，则不需要处理
        if (CollectionUtils.isEmpty(originals)) {
            return;
        }
        // 【记忆变量】对话缓存不存在，则不需要处理
        Map<Object, Object> memoryVariable = MemoryVariableWriter.get(identifier);
        if (MapUtils.isEmpty(memoryVariable)) {
            return;
        }

        // 1. 获取需要删除的字段集合
        // 2. 获取需要更新值的字段集合
        Set<String> delKeys = new HashSet<>();
        Set<String> updateKeys = new HashSet<>();
        for (Variable variable : originals) {
            String key = variable.getKey();
            boolean needDel = transformed.stream().noneMatch(v -> v.getKey().equals(key));
            if (needDel) {
                delKeys.add(key);
            }

            boolean needUpdate = transformed.stream().anyMatch(v -> v.getKey().equals(key));
            if (needUpdate) {
                Optional<Variable> target = transformed.stream().filter(v -> v.getKey().equals(key)).findFirst();
                if (StringUtils.isBlank(target.get().getVariableDefault())) {
                    continue;
                }
                Map<Object, Object> map = MemoryVariableWriter.get(identifier);
                if (map == null) {
                    map = new HashMap<>();
                }
                Object value = map.get(key);
                if (value == null || StringUtils.isBlank(value.toString())) {
                    updateKeys.add(key);
                }
            }
        }
        // 3. 获取需要新增的字段集合
        Set<String> addKeys = new HashSet<>();
        for (Variable variable : transformed) {
            String key = variable.getKey();
            boolean needAdd = originals.stream().noneMatch(v -> v.getKey().equals(key));
            if (needAdd) {
                addKeys.add(key);
            }
        }

        // 删除
        if (!delKeys.isEmpty()) {
            MemoryVariableWriter.delItem(identifier, delKeys.toArray(new String[0]));
        }
        // 新增
        if (!addKeys.isEmpty()) {
            Map<String, Object> map = new HashMap<>();
            for (String key : addKeys) {
                Variable variable = transformed.stream().filter(v -> v.getKey().equals(key)).findFirst().get();
                map.put(variable.getKey(), variable.getVariableDefault());
            }
            MemoryVariableWriter.addItem(identifier, map);
        }
        // 更新
        if (!updateKeys.isEmpty()) {
            Map<String, Object> map = new HashMap<>();
            for (String key : updateKeys) {
                Variable variable = transformed.stream().filter(v -> v.getKey().equals(key)).findFirst().get();
                map.put(variable.getKey(), variable.getVariableDefault());
            }
            MemoryVariableWriter.addItem(identifier, map);
        }
    }
}
