import { NEllipsis, NSwitch } from 'naive-ui'
import { KnowledgeDocumentItem, KnowledgeItem } from './types.d'
import { formatDateTime } from '@/utils/date-formatter'
import i18n from '@/locales'
import { KnowledgeTypeIcon } from '@/enums/knowledge'

const t = i18n.global.t

export function createKnowledgeColumn(
  handleKnowledgeTableAction: (actionType: string, knowledgeId: number, KnowledgeItem?: KnowledgeItem) => void,
) {
  return [
    {
      title: () => <span>{t('personal_space_module.knowledge_module.knowledge_name')}</span>,
      key: 'knowledgeName',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 210,
      fixed: 'left',
      render(row: KnowledgeItem) {
        return (
          <div class='flex items-center gap-[10px]'>
            <div
              class='h-[22px] w-[22px] flex-shrink-0 bg-contain bg-no-repeat'
              style={{ backgroundImage: `url(${KnowledgeTypeIcon[row.knowledgeType]})` }}
            />
            <NEllipsis class='flex-1'>{row.knowledgeName || '--'}</NEllipsis>
          </div>
        )
      },
    },
    {
      title: () => <span>{t('personal_space_module.knowledge_module.knowledge_desc')}</span>,
      key: 'knowledgeDesc',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 380,
      render(row: KnowledgeItem) {
        return row.desc || '--'
      },
    },
    {
      title: () => <span>{t('common_module.data_table_module.type')}</span>,
      key: 'type',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 170,
      render(row: KnowledgeItem) {
        const knowledgeTypeMap = {
          Base: t('personal_space_module.knowledge_module.base_knowledge_type'),
          QA: t('personal_space_module.knowledge_module.QA_knowledge_type'),
        }

        return knowledgeTypeMap?.[row.knowledgeType] || '--'
      },
    },
    {
      title: () => <span>{t('common_module.created_time')}</span>,
      key: 'createdTime',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 170,
      render(row: KnowledgeItem) {
        return row.createdTime ? formatDateTime(row.createdTime) : '--'
      },
    },
    {
      title: () => <span>{t('common_module.is_open')}</span>,
      key: 'isOpen',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 150,
      render(row: KnowledgeItem) {
        return (
          <NSwitch
            value={row.isOpen === 'Y'}
            onUpdateValue={() => handleKnowledgeTableAction('updateOpen', row.id, row)}
          />
        )
      },
    },
    {
      title: () => <span>{t('common_module.data_table_module.action')}</span>,
      key: 'action',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 190,
      fixed: 'right',
      render(row: KnowledgeItem) {
        return (
          <div>
            <span
              className='text-theme-color mr-5 cursor-pointer hover:opacity-80'
              onClick={() => handleKnowledgeTableAction('view', row.id)}
            >
              {t('common_module.data_table_module.view')}
            </span>
            <span
              className='text-error-font-color mr-5 cursor-pointer hover:opacity-80'
              onClick={() => handleKnowledgeTableAction('delete', row.id)}
            >
              {t('common_module.data_table_module.delete')}
            </span>
          </div>
        )
      },
    },
  ]
}

export function createKnowledgeDocumentColumn(
  handleKnowledgeDocumentTableAction: (actionType: string, knowledgeDocumentItem: KnowledgeDocumentItem) => void,
) {
  return [
    {
      type: 'selection',
      fixed: 'left',
      disabled(row: KnowledgeDocumentItem) {
        return !['Complete', 'Fail'].includes(row.trainStatus)
      },
    },
    {
      title: () => <span>{t('personal_space_module.knowledge_module.knowledge_document_name')}</span>,
      key: 'documentName',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 316,
      render(row: KnowledgeDocumentItem) {
        return row.documentName || '--'
      },
    },
    {
      title: () => <span>{t('personal_space_module.knowledge_module.knowledge_document_char_count')}</span>,
      key: 'charCount',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 222,
      render(row: KnowledgeDocumentItem) {
        return `${row.charCount || 0}` + t('common_module.char')
      },
    },
    {
      title: () => <span>{t('personal_space_module.knowledge_module.knowledge_document_format')}</span>,
      key: 'format',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 236,
      render(row: KnowledgeDocumentItem) {
        return row.documentName.split('.')?.pop()?.toLowerCase() || '--'
      },
    },
    {
      title: () => <span>{t('common_module.modified_time')}</span>,
      key: 'uploadTime',
      align: 'left',
      ellipsis: {
        tooltip: true,
      },
      width: 250,
      render(row: KnowledgeDocumentItem) {
        return formatDateTime(row.uploadTime) || '--'
      },
    },
    {
      title: () => <span>{t('common_module.is_open')}</span>,
      key: 'trainStatus',
      align: 'left',
      width: 224,
      render(row: KnowledgeDocumentItem) {
        const trainStatusMap = {
          Unopened: { text: 'common_module.studying', color: '#0B7DFF' },
          Line: { text: 'common_module.studying', color: '#0B7DFF' },
          Training: { text: 'common_module.studying', color: '#0B7DFF' },
          Complete: { text: 'common_module.available', color: '#0DD623' },
          Fail: { text: 'common_module.unavailable', color: '#F25744' },
        }
        const state = trainStatusMap[row.trainStatus]

        return (
          <div class='flex items-center'>
            {['Unopened', 'Line', 'Training'].includes(row.trainStatus) && (
              <i class='iconfont icon-study mr-1.5 text-[#0B7DFF]' />
            )}

            {['Complete', 'Fail'].includes(row.trainStatus) && (
              <div class='mr-2.5 h-3 w-3 rounded-full' style={{ backgroundColor: state.color }} />
            )}

            <span>{t(state.text)}</span>
          </div>
        )
      },
    },
    {
      title: () => <span>{t('common_module.data_table_module.action')}</span>,
      key: 'action',
      align: 'left',
      width: 273,
      fixed: 'right',
      render(row: KnowledgeDocumentItem) {
        return (
          <div>
            {row.trainStatus !== 'Fail' && (
              <span
                class={[
                  row.trainStatus === 'Complete'
                    ? 'mr-5 cursor-pointer text-[#0B7DFF] hover:opacity-80'
                    : 'text-gray-font-color pointer-events-none mr-5 cursor-not-allowed',
                ]}
                onClick={() => handleKnowledgeDocumentTableAction('view', row)}
              >
                {t('personal_space_module.knowledge_module.view_knowledge_document_fragment')}
              </span>
            )}

            {row.trainStatus === 'Fail' && (
              <span
                class='mr-5 cursor-pointer text-[#0B7DFF] hover:opacity-80'
                onClick={() => handleKnowledgeDocumentTableAction('train', row)}
              >
                {t('personal_space_module.knowledge_module.training_knowledge_document')}
              </span>
            )}

            <span
              class={[
                ['Complete', 'Fail'].includes(row.trainStatus)
                  ? 'text-error-font-color mr-5 cursor-pointer hover:opacity-80'
                  : 'text-gray-font-color pointer-events-none mr-5 cursor-not-allowed',
              ]}
              onClick={() => handleKnowledgeDocumentTableAction('delete', row)}
            >
              {t('common_module.data_table_module.delete')}
            </span>
          </div>
        )
      },
    },
  ]
}
