import { request } from '@/utils/request'

/**
 *
 * @param payload agentApplicationInfo 应用参数
 * @returns 新建或更新应用
 */
export function fetchSaveAgentApplication<T>(payload: object) {
  return request.post<T>('/agentApplicationInfoRest/saveOrUpdate.json', payload)
}

/**
 *
 * @param payload 查询参数
 * @returns 获取应用列表
 */
export function fetchGetApplicationList<T>(payload: object) {
  return request.post<T>('/agentApplicationInfoRest/getListByMember.json', payload)
}

/**
 *
 * @param payload agentId 应用Id
 * @returns 通过agentId删除应用
 */
export function fetchDeleteApplication<T>(agentId: string) {
  return request.post<T>(`/agentApplicationInfoRest/delete.json?agentId=${agentId}`)
}

/**
 *
 * @param payload agentId 应用Id
 * @returns 通过agentId获取调试应用详情
 */
export function fetchGetDebugApplicationInfo<T>(agentId: string) {
  return request.post<T>(`/agentApplicationInfoRest/getInfo.json?agentId=${agentId}`)
}

/**
 *
 * @param payload agentId 应用Id
 * @returns 通过agentId获取发布应用详情
 */
export function fetchGetApplicationInfo<T>(agentId: string) {
  return request.post<T>(`/agentApplicationRest/getInfo.json?agentId=${agentId}`)
}

/**
 *
 * @param payload payload 应用参数
 * @returns 发布应用
 */
export function fetchPublishApplication<T>(payload: object) {
  return request.post<T>('/agentApplicationInfoRest/updateAndPublish.json', payload)
}

/**
 * @returns 获取大模型列表
 */
export function fetchGetLargeModelList<T>() {
  return request.post<T>('/agentApplicationInfoRest/getLargeModelListV2.json')
}

/**
 * * @param agentId 应用Id
 * @returns 创建会话Id
 */
export function fetchCreateDialogues<T>(agentId: string) {
  return request.post<T>(`/agentApplicationRest/createDialogues.json?agentId=${agentId}`)
}

/**
 * * @param  { input: AI回答内容 }
 * @returns 生成追问问题
 */
export function fetchCreateContinueQuestions<T>(payload: { input: string }) {
  return request.post<T>('/agentApplicationRest/createContinueQuestions.json', payload)
}

/**
 * * @param { payload: { agentTitle: 标题 agentDesc: 描述 }, controller: 控制取消请求 }
 * @returns AI生成应用头像
 */
export function fetchCreateAgentAvatar<T>(
  payload: { agentTitle: string; agentDesc: string },
  controller: AbortController,
) {
  return request.post<T>('/agentApplicationInfoRest/createAgentApplicationAvatar.json', payload, {
    timeout: 120000,
    signal: controller.signal,
  })
}

/**
 * * @param { payload: { agentTitle: 标题, agentDesc: 描述, agentSystem: 指令 }, controller: 控制取消请求 }
 * @returns AI生成开场白
 */
export function fetchCreatePreamble<T>(
  payload: { agentTitle: string; agentDesc: string; agentSystem: string },
  controller: AbortController,
) {
  return request.post<T>('/agentApplicationInfoRest/createPreamble.json', payload, {
    timeout: 120000,
    signal: controller.signal,
  })
}

/**
 * * @param { payload: { agentTitle: 标题 agentDesc: 描述 }, controller: 控制取消请求 }
 * @returns AI生成推荐问
 */
export function fetchCreateFeaturedQuestions<T>(
  payload: { agentTitle: string; agentDesc: string },
  controller: AbortController,
) {
  return request.post<T>('/agentApplicationInfoRest/createFeaturedQuestions.json', payload, {
    timeout: 120000,
    signal: controller.signal,
  })
}

/**
 * * @param { payload: { input: 自动配置信息 }, controller: 控制取消请求 }
 * @returns 生成应用标题和描述
 */
export function fetchCreateAgentTitleAndDesc<T>(payload: { input: string }, controller: AbortController) {
  return request.post<T>('/agentApplicationInfoRest/createAgentTitleAndDesc.json', payload, {
    timeout: 120000,
    signal: controller.signal,
  })
}

/**
 * * @param { modelName: 模型名称 }
 * @returns 搜索模型信息
 */
export function fetchGetLargeModelInfo<T>(modelName: string) {
  return request.post<T>(`/agentApplicationInfoRest/getLargeModelInfoV2.json?query=${modelName}`)
}

/**
 * * @param agentId 应用Id
 * @returns 收藏应用
 */
export function fetchApplicationsCollectionStatusChange<T>(agentId: string) {
  return request.post<T>(`/agentApplicationInfoRest/collectOrCancelAgentInPerson.json?agentId=${agentId}`)
}

/**
 *
 * @param payload payload 应用参数
 * @returns 上架应用
 */
export function fetchSalePublishApplication<T>(payload: object) {
  return request.post<T>('/bizAgentApplicationMallRest/publishAgentToMall.json', payload)
}

/**
 * * @param agentPublishId 发布应用表的id
 * @returns 下架应用
 */
export function fetchRemoveSalePublishApplication<T>(agentPublishId: number) {
  return request.post<T>(`/bizAgentApplicationMallRest/unSaleAgentInMall.json?agentPublishId=${agentPublishId}`)
}

/**
 * * @param agentPublishId 发布应用表的id
 * @returns 已上架应用信息
 */
export function fetchGetApplicationMallInfo<T>(agentId: string) {
  return request.post<T>(`/bizAgentApplicationMallRest/getMallInfoByAgentId.json?agentId=${agentId}`)
}

/**
 * @query agentId 应用Id
 * @returns 获取用户自动播放配置
 */
export function fetchGetAutoPlayByAgentId<T>(agentId: string) {
  return request.post<T>(`/agentApplicationRest/autoPlayByAgentId.json?agentId=${agentId}`)
}

/**
 * @query agentId 应用Id
 * @query autoPlay 是否自动播放
 * @returns 设置用户自动播放配置
 */
export function fetchUpdateAutoPlay<T>(agentId: string, autoPlay: 'Y' | 'N') {
  return request.post<T>(`/agentApplicationRest/enableAutoPlay.json?agentId=${agentId}&autoPlay=${autoPlay}`)
}

/**
 * @returns 获取API配置
 */
export function fetchGetAPIProfile<T>() {
  return request.post<T>('/agentApplicationRest/getApiProfile.json')
}
