import { nanoid } from 'nanoid/non-secure'
import { createWebHashHistory, createWebHistory, type RouteRecordRaw, type RouterHistory } from 'vue-router'

export interface MenuOption {
  // label: (() => VNode) | string
  label: string
  key: string
  routeName: string
  icon: string
  children?: MenuOption[]
}

export function getHistoryMode(modeString: ViteEnv['VITE_ROUTER_MODE']): RouterHistory {
  const PUBLIC_PATH = import.meta.env.VITE_PUBLIC_PATH

  if (modeString === 'h5') {
    return createWebHistory(PUBLIC_PATH)
  }

  return createWebHashHistory(PUBLIC_PATH)
}

function menuSort(routes: RouteRecordRaw[]) {
  routes.sort((a, b) => {
    if (!a.meta || !a.meta.rank) {
      return 1
    }

    if (!b.meta || !b.meta.rank) {
      return -1
    }

    return (a.meta.rank as number) - (b.meta.rank as number)
  })

  return routes
}

export function menuFilterSort(routes: RouteRecordRaw[]) {
  function createRouteKey(routes: RouteRecordRaw[]) {
    routes.forEach((route, index) => {
      route.meta ? (route.meta.key = nanoid()) : (route.meta = { key: nanoid(), title: '', rank: 1001 + index })

      if (route.children) {
        createRouteKey(route.children)
      }
    })
  }
  createRouteKey(routes)

  function menuChildrenSort(routes: RouteRecordRaw[]) {
    routes.forEach((routeItem) => {
      if (routeItem.children) {
        const newRouteChildren = menuSort(routeItem.children)
        routeItem.children = newRouteChildren

        menuChildrenSort(routeItem.children)
      }
    })
  }
  menuChildrenSort(routes)
  menuSort(routes)

  const rootRouteIndex = routes.findIndex((route) => route.name === 'Root')

  if (rootRouteIndex >= 0) {
    const rootRouteChildren = routes[rootRouteIndex].children || []
    routes.splice(rootRouteIndex, 1)
    routes.unshift(...rootRouteChildren)
  }

  function createMenuOptions(routes: RouteRecordRaw[]) {
    const menuOptions: MenuOption[] = []

    routes.forEach((route) => {
      // 菜单Item隐藏判断
      if (route.meta?.hideSideMenItem) {
        return
      }

      const menuOption: MenuOption = {
        // label: route.meta?.title
        //   ? () => h(RouterLink, { to: { name: route.name || 'Root' } }, { default: () => route.meta?.title })
        //   : '-',
        label: (route.meta?.title as string) || '-',
        key: route.meta?.key as string,
        routeName: route.name as string,
        icon: route.meta?.icon as string,
      }

      if (route.children) {
        menuOption.children = createMenuOptions(route.children)
      }

      menuOptions.push(menuOption)
    })

    return menuOptions
  }

  return createMenuOptions(routes)
}
