import { request } from '@/utils/request'
import qs from 'qs'

/* 知识库部分开始 */

/**
 * @query trainStatus 训练状态
 * @query search 搜索值
 * @returns 获取知识库列表
 */
export function fetchGetKnowledgeList<T>(trainStatus: string, search: string, payload: object) {
  return request.post<T>(`/knowledgeRest/getKnowledgeList.json?trainStatus=${trainStatus}&search=${search}`, payload)
}

/**
 * @param payload { decs: string, knowledgeName: string }
 * @returns 新建知识库
 */
export function fetchCreateKnowledge<T>(payload: object) {
  return request.post<T>('/knowledgeRest/createKnowledge.json', payload)
}

/**
 * @param payload { id：number, decs: string, knowledgeName: string }
 * @returns 更新知识库
 */
export function fetchUpdateKnowledgeInfo<T>(payload: object) {
  return request.post<T>('/knowledgeRest/updateKnowledgeInfo.json', payload)
}

/**
 * @query knowledgeInfoId 知识库Id
 * @returns 通过Id获取知识库详情
 */
export function fetchGetKnowledgeDetail<T>(knowledgeInfoId: number) {
  return request.post<T>(`/knowledgeRest/getKnowledgeDetail.json?knowledgeInfoId=${knowledgeInfoId}`)
}

/**
 * @query knowledgeInfoId 知识库Id
 * @query isOpen 是否开启 Y | N
 * @returns 开关知识库
 */
export function fetchEnableKnowledgeInfo(knowledgeInfoId: number, isOpen: string) {
  return request.post(`/knowledgeRest/enableKnowledgeInfo.json?knowledgeInfoId=${knowledgeInfoId}&isOpen=${isOpen}`)
}

/**
 * @query knowledgeInfoId 知识库Id
 * @returns 批量删除知识库
 */
export function fetchDelKnowledgeById<T>(knowledgeInfoId: number) {
  return request.post<T>(`/knowledgeRest/deleteKnowledgeInfo.json?knowledgeInfoId=${knowledgeInfoId}`)
}

/**
 * @query formData 上传文档内容
 * @returns 上传知识库文档
 */
export function fetchUploadKnowledgeDocument<T>(formData: FormData) {
  return request.post<T>('/knowledgeRest/uploadDocument.json', formData, {
    headers: {
      'Content-Type': 'multipart/form-data',
    },
    timeout: 300000,
  })
}

/**
 * @params { knowledgeInfoId: number, kdIds: number[], segmentationConfig: SegmentationConfigInterface }
 * @returns 训练知识库
 */
export function fetchTrainKnowledge<T>(payload: object) {
  return request.post<T>('/knowledgeRest/trainKnowledge.json', payload)
}

/**
 * @params kdIds 知识库Id列表
 * @returns 【根据知识库ID】获取知识库列表
 */
export function fetchGetKnowledgeListByKdIds<T>(kdIds: number[]) {
  return request.post<T>('/knowledgeRest/getKnowledgeListByKnowledgeInfoIds.json', kdIds)
}

/* 知识库部分结束 */

/* 知识库文档部分开始 */

/**
 * @query knowledgeInfoId 知识库Id
 * @query search 搜索值
 * @returns 获取知识库文档列表
 */
export function fetchGetKnowledgeDocumentList<T>(knowledgeInfoId: number, search: string) {
  return request.post<T>(`/knowledgeRest/searchDocuments.json?knowledgeInfoId=${knowledgeInfoId}&search=${search}`)
}

/**
 * @query knowledgeInfoId 知识库Id
 * @params kdIds 知识库文档Id
 * @returns 批量删除知识库文档
 */
export function fetchBatchDelKnowledgeDocument(knowledgeInfoId: number, payload: number[]) {
  return request.post(`/knowledgeRest/batchDelDocument.json?knowledgeInfoId=${knowledgeInfoId}`, payload)
}

/**
 * @query knowledgeInfoId 知识库Id
 * @query kdId 知识库文档Id
 * @returns 删除知识库文档
 */
export function fetchDelKnowledgeDocument(knowledgeInfoId: number, kdId: number) {
  return request.post(`/knowledgeRest/delDocument.json?knowledgeInfoId=${knowledgeInfoId}&kdId=${kdId}`)
}

/**
 * @params kdIds 知识库文档Id
 * @returns 根据kdIds获取知识库文档列表
 */
export function fetchGetKnowledgeDocumentListByKdIds<T>(kdIds: number[]) {
  return request.post<T>(`/knowledgeRest/getListByKdIds.json`, kdIds)
}

/* 知识库文档片段开始 */

/**
 * @query query 模糊搜索
 * @query kdId 知识库文档Id
 * @params payload { page: number, pageSize: number }
 * @returns 获取知识库分片列表
 */
export function fetchGetKnowledgeChunkList<T>(query: string, kdId: number, payload: object) {
  return request.post<T>(`/knowledgeRest/getChunks.json?query=${query}&kdIds=${kdId}`, payload)
}

/**
 * @params payload { kdId: number, chunkContent: string, chunkSort: number }
 * @returns 新增知识库分片
 */
export function fetchAddKnowledgeChunk<T>(payload: { kdId: number; chunkContent: string; chunkSort: number }) {
  return request.post<T>('/knowledgeRest/addKnowledgeChunk.json', payload)
}

/**
 * @params payload { kdId: number, chunkContent: string, chunkRelationId: string }
 * @returns 更新知识库分片
 */
export function fetchUpdateKnowledgeChunk<T>(payload: { kdId: number; chunkContent: string; chunkRelationId: string }) {
  return request.post<T>('/knowledgeRest/updateKnowledgeChunkDoc.json', payload)
}

/**
 * @params payload { kdId: number, chunkRelationId: string }
 * @returns 删除知识库分片
 */
export function fetchDeleteKnowledgeChunk<T>(payload: { kdId: number; chunkRelationId: string }) {
  return request.post<T>('/knowledgeRest/deleteKnowledgeChunk.json', payload)
}

/**
 * @params payload { kdId: number, chunkRelationId: string, isOpen: 'Y' | 'N' }
 * @returns 开关知识库分片
 */
export function fetchOpenKnowledgeChunk<T>(payload: { kdId: number; chunkRelationId: string; isOpen: 'Y' | 'N' }) {
  return request.post<T>('/knowledgeRest/openKnowledgeChunk.json', payload)
}
/* 知识库文档部分结束 */

/* 问答知识库文档片段开始 */

/**
 * @query kdId 知识库文档Id
 * @returns 获取问答知识库分片结构
 */
export function fetchQAKnowledgeChunkStruct<T>(kdId: number) {
  return request.post<T>(`/qAKnowledgeRest/getKnowledgeStruct.json?kdId=${kdId}`)
}

/**
 * @query query 模糊搜索
 * @query kdId 知识库文档Id
 * @params payload { page: number, pageSize: number }
 * @returns 获取问答知识库分片列表
 */
export function fetchQAKnowledgeChunkList<T>(query: string, kdId: number, payload: object) {
  return request.post<T>(`/qAKnowledgeRest/getQAKnowledgeChunks.json?query=${query}&kdId=${kdId + ''}`, payload)
}

/**
 * @query kdId 知识库文档Id  structId 结构Id structName 结构名称 isIndex 是否开启索引
 * @returns 修改问答知识库表结构
 */
export function fetchUpdateQAKnowledgeStruct<T>(payload: object) {
  return request.post<T>(
    '/qAKnowledgeRest/updateKnowledgeChunk.json',
    {},
    {
      params: payload,
      paramsSerializer: function (params) {
        return qs.stringify(params, { arrayFormat: 'repeat' })
      },
    },
  )
}

/**
 * @params { kdId 知识库文档Id chunkInfos 分片内容 chunkSort 文档顺序 }
 * @returns 添加问答知识库分片
 */
export function fetchAddQAKnowledgeChunk<T>(payload: object) {
  return request.post<T>('/qAKnowledgeRest/addKnowledgeChunk.json', payload)
}

/**
 * @params { kdId 知识库文档Id chunkRelationId 分片Id chunkInfos 分片内容 chunkSort 文档顺序 }
 * @returns 编辑问答知识库分片
 */
export function fetchEditQAKnowledgeChunk<T>(payload: object) {
  return request.post<T>('/qAKnowledgeRest/batchUpdateKnowledgeChunkDoc.json', payload, {
    timeout: 15000,
  })
}

/**
 * @payload kdId 知识库文档Id  chunkRelationId 分片Id
 * @returns 删除问答知识库分片
 */
export function fetchDeleteQAKnowledgeChunk<T>(payload: { kdId: number; chunkRelationId: string }) {
  return request.post<T>('/qAKnowledgeRest/deleteKnowledgeChunk.json', payload)
}

/**
 * @payload kdId 知识库文档Id  chunkRelationId 分片Id isOpen 开启状态
 * @returns 问答知识库分片开启
 */
export function fetchOpenQAKnowledgeChunk<T>(payload: { kdId: number; chunkRelationId: string; isOpen: 'Y' | 'N' }) {
  return request.post<T>('/qAKnowledgeRest/openKnowledgeChunk.json', payload)
}

/**
 * @query kdId 知识库文档Id  chunkRelationIds 分片Id数组
 * @returns 批量删除问答知识库分片
 */
export function fetchBatchDeleteQAKnowledgeChunks<T>(kdId: number, chunkRelationIds: string[]) {
  return request.post<T>(
    `/qAKnowledgeRest/batchDeleteKnowledgeChunks.json?kdId=${kdId}&chunkRelationIds=${chunkRelationIds}`,
    null,
    {
      timeout: 40000,
    },
  )
}

/* 知识库文档部分结束 */
